﻿/***************************************************************************
 * This software written by Ufasoft  http://www.ufasoft.com                *
 * It is Public Domain and can be used in any Free or Commercial projects  *
 * with keeping these License lines in Help, Documentation, About Dialogs  *
 * and Source Code files, derived from this.                               *
 * ************************************************************************/

using System;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.Net.Mail;
using System.Collections.Specialized;

namespace Utils {

	public class LineServerException : ApplicationException {
		public LineServerException(string s)
			:	base(s)
			{}

	}

	public class LineServer {
		public TextReader Reader;
		public TextWriter Writer;

		protected bool Exit;

		protected virtual void OnLine(string line) {}
		protected virtual void OnRun() { }

		public void WriteLine(string s, params object[] arg) {
			Writer.WriteLine(s, arg);
			Writer.Flush();
		}

		public void Run() {
			Writer.NewLine = "\r\n"; // <CR><LF>  in Internet Protocols
			OnRun();
			for (string line; (line = Reader.ReadLine()) != null; ) {
				OnLine(line);
				if (Exit)
					break;
			}
			//Writer.Flush();
			Writer.Close();
		}
	}

	public class CommandLineServer : LineServer {

		static Regex s_reCmd = new Regex(@"^([a-zA-Z]+)(\s+(.*))?", RegexOptions.Compiled);

		protected virtual void OnCommand(string cmd, string args) {
		}

		protected virtual void OnNotCommand() {
			throw new LineServerException("Not command");
		}

		protected override void OnLine(string line) {
			Match m = s_reCmd.Match(line);
			if (m.Success) {
				string cmd = m.Groups[1].Value.ToUpper();
				string args = m.Groups[3].Value;
				OnCommand(cmd, args);
			}
			else
				OnNotCommand();
		}
	}

	public class MailMessageEx : MailMessage { //!!! workaroundto fix Mono bug
		public new NameValueCollection Headers = new NameValueCollection();
		public new string Subject {
			get {
				return Headers["Subject"] != null ? Headers["Subject"] : "";
			}
		}
	}

	public class InetCLServer : CommandLineServer {

		public delegate void CommandHandler(string cmd, string args);

		protected Dictionary<string, CommandHandler> CommandSet = new Dictionary<string, CommandHandler>();

		protected override void OnCommand(string cmd, string args) {
			CommandHandler handler = CommandSet[cmd];
			try {
				if (handler == null)
					throw new SmtpException(SmtpStatusCode.CommandUnrecognized);
				handler(cmd, args);
			}
			catch (SmtpException x) {
				WriteLine("{0} {1}", (int)x.StatusCode, x.Message);
			}
			catch (Exception x) {
				WriteLine("{0} {1}", (int)SmtpStatusCode.GeneralFailure, x.Message);
			}
		}

		protected void EndMultiline() {
			WriteLine(".");
		}

		protected void WriteLines(IEnumerable<string> lines) {
			foreach (string line in lines) {
				string s = line;
				if (line.Length>0 && line[0] == '.')
					s = "." + line;
				Writer.WriteLine(s);
			}
			EndMultiline();
		}

		protected MailMessageEx ReadMessage() {
			var mm = new MailMessageEx();
			Ut.ReadMailHeader(mm.Headers, Reader);
			var sb = new StringBuilder();
			for (string line; (line = Reader.ReadLine()) != "."; ) {
				if (line.Length > 0 && line[0] == '.')
					line = line.Substring(1);
				sb.Append(line + "\n");
			}
			mm.Body = sb.ToString();
			return mm;
		}

	}

	public class LineProtocolLogger {

		class LPReader : TextReader {
			internal LineProtocolLogger logger;
			internal TextReader r;

			public override int Peek() {
				int ch = r.Peek();
				return ch;
			}
			
			public override int Read() {
				int ch = r.Read();
				if (ch != -1)
					logger.WriteLog((char)ch, false);
				return ch;
			}

			public override string ReadLine() {
				string line = r.ReadLine();
				foreach (char ch in line)
					logger.WriteLog(ch, false);
				logger.WriteLog('\n', false);
				return line;
			}
		}

		class LPWriter : TextWriter {
			internal LineProtocolLogger logger;
			internal TextWriter w;

			public override Encoding Encoding { get { return w.Encoding; } }

			public override void Flush() {
				w.Flush();
				logger.logWriter.Flush();
			}

			public override void Write(char ch) {
				w.Write(ch);
				logger.WriteLog(ch, true);
			}
		}

		public string Dir = "/var/log";

		public TextReader Reader;
		public TextWriter Writer;

		internal StreamWriter logWriter;

		public LineProtocolLogger(TextReader r, TextWriter w) {
			var lpr = new LPReader();
			lpr.logger = this;
			lpr.r = r;
			Reader = lpr;

			var lpw = new LPWriter();
			lpw.logger = this;
			lpw.w = w;
			Writer = lpw;
		}

		bool bAtBOL = true;
		bool bPrevOut = false;

		void WriteLog(char ch, bool bOut) {
			if (logWriter == null) {
				Directory.CreateDirectory(Dir);
				Random random = new Random((int)DateTime.Now.Ticks);
				string filename = Path.Combine(Dir, random.Next(10).ToString());
				File.Delete(filename);
				logWriter = new StreamWriter(filename);
				logWriter.AutoFlush = true; //!!!D
			}
			if (bPrevOut != bOut && !bAtBOL) {
				logWriter.Write('\n');
				bAtBOL = true;
			}
			if (bAtBOL) {
				logWriter.Write(bOut ? "< " : "> ");
				bAtBOL = false;
			}
			if (ch == '\n') {
				if (bOut == bPrevOut)
					logWriter.Write(ch);
				bAtBOL = true;
			}
			else {				
				logWriter.Write(ch);
			}
			bPrevOut = bOut;
		}

	}
}



