﻿/***************************************************************************
 * This software written by Ufasoft  http://www.ufasoft.com                *
 * It is Public Domain and can be used in any Free or Commercial projects  *
 * with keeping these License lines in Help, Documentation, About Dialogs  *
 * and Source Code files, derived from this.                               *
 * ************************************************************************/

// NNTP-server implementation Base class
// based on RFC 977, RFC 2980

using System;
using System.IO;
using System.Collections.Specialized;
using System.Collections.Generic;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;
using System.Net.Mail;


namespace Utils {

	public enum NntpStatusCode : ushort {
		ServiceDiscontinued		= 400,
		NoSuchGroup				= 411,
		NoGroupSelected			= 412,
		NoArticleSelected		= 420,
		NoNextArticle			= 421,
		NoPrevArticle			= 422,
		NoSuchArticle			= 423,
		NoSuchArticleFound		= 430,
		ArticleNotWanted		= 435,
		TransferFailed			= 436,
		ArticleRejected			= 437,
		PostingNotAllowed		= 440,
		PostingFailed			= 441,
		AuthRequired 			= 480,
		AuthRejected 			= 482,
		CommandNotRecognized	= 500,
		CommandSyntaxError		= 501,
		PermissionDenied		= 502,
		ProgramFault			= 503
	}

	public class NntpException : SmtpException {
		public NntpException(NntpStatusCode sc)
			:	base((SmtpStatusCode)sc)
		{}

		public new NntpStatusCode StatusCode {
			get { return (NntpStatusCode)base.StatusCode; }
		}

		public override string Message {
			get {
				switch (StatusCode) {
					case NntpStatusCode.ServiceDiscontinued:	return "service discontinued";
					case NntpStatusCode.NoSuchGroup:			return "no such news group";
					case NntpStatusCode.NoGroupSelected:		return "no newsgroup has been selected";
					case NntpStatusCode.NoArticleSelected:		return "no current article has been selected";
					case NntpStatusCode.NoNextArticle:			return "no next article in this group";
					case NntpStatusCode.NoPrevArticle:			return "no previous article in this group";
					case NntpStatusCode.NoSuchArticle:			return "no such article number in this group";
					case NntpStatusCode.NoSuchArticleFound:		return "no such article found";
					case NntpStatusCode.ArticleNotWanted:		return "article not wanted - do not send it";
					case NntpStatusCode.TransferFailed:			return "transfer failed - try again later";
					case NntpStatusCode.ArticleRejected:		return "article rejected - do not try again";
					case NntpStatusCode.PostingNotAllowed:		return "posting not allowed";
					case NntpStatusCode.PostingFailed:			return "posting failed";
					case NntpStatusCode.AuthRequired:			return "Authentication required";
					case NntpStatusCode.AuthRejected:			return "Authentication rejected";
					case NntpStatusCode.CommandNotRecognized:	return "command not recognized";
					case NntpStatusCode.CommandSyntaxError:		return "command syntax error";
					case NntpStatusCode.PermissionDenied:		return "access restriction or permission denied";
					case NntpStatusCode.ProgramFault:			return "program fault - command not performed";
					default:
						return "Unknown status code";
				}
			}
		}
	}

	public class Newsgroup {
		public NntpServer NntpServer;

		public int Id;
		public string Name;
		public int First,
			       Last,
				   EstimatedCount;

		public List<int> ArticleIds {
			get { return NntpServer.GetArticleIds(Name); }
		}
	};

	public class NewsArticle : MailMessage {
		public Newsgroup Newsgroup;
		public int Id;
		public string Author;
		public DateTime Date;
		public string MessageId;
		public List<string> Refs = new List<string>();
		public int ByteCount;
		public int LineCount;

		static string HostName = Dns.GetHostName();

		public new NameValueCollection Headers {
			get {
				var headers = new NameValueCollection();
				headers["From"] = Rfc822Message.GetEncoded(Author);
				headers["Subject"] = Rfc822Message.GetEncoded(Subject);

#if DEBUG
				Console.Error.WriteLine("BeforeRefs");
#endif


				if (Refs.Count != 0) {
					headers["Subject"] = "Re: " + Rfc822Message.GetEncoded(Subject);
					var r = new StringBuilder();
					foreach (var s in Refs)
						r.Append(string.Format("<{0}> ", s));
					headers["References"] = r.ToString();
				}

#if DEBUG
				Console.Error.WriteLine("AfterRefs");
#endif


				headers["Message-ID"] = "<" + MessageId + ">";
				headers["Date"] = Date.ToString("r");
				headers["Newsgroups"] = Newsgroup.Name;
				headers["Path"] = HostName+"!not-for-mail";
				headers["Lines"] = LineCount.ToString();
				headers["Xref"] = HostName.Split('.')[0] + " " + Newsgroup.Name + ":" + Id.ToString();
				if (!Ut.IsAscii(Body))
					headers["Content-type"] = "text/plain; charset=utf-8";

#if DEBUG
				Console.Error.WriteLine("EndHeaders");
#endif

				return headers;
			}
		}

		static string EscapeQuotes(string s) {
			return s.Replace("\"", "\\\"");
		}

		public string GetOverview()
		{
			var r = new StringBuilder();
			r.Append(string.Format("{0}\t\"{1}\"\t\"{2}\"\t{3}\t<{3}>\t", Id, EscapeQuotes(Subject), EscapeQuotes(Author),
				Date.ToString("r"), MessageId));
			foreach (var s in Refs)
				r.Append(string.Format("<{0}> ", s));
			r.Append(string.Format("\t{0}\t{1}", ByteCount, LineCount));
			return r.ToString();
		}	
	};

	public class NewsUser {
		public int Id;
		public string Login, Password;
	}

	public class NntpServer : InetCLServer {

		public string Promt;
		public bool ReadOnly = true;
		public bool AuthRequired = false;
		public Newsgroup SelectedGroup;
		public NewsArticle SelectedArticle;
		public NewsUser User;
		public string Login, Password;

		public virtual List<int> GetArticleIds(string groupname) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected virtual IEnumerable<string> GetLIST() {
			return new string[] { };
		}

		protected virtual IEnumerable<string> GetGroupsSince(DateTime dt) {
			return GetLIST();
		}

		protected virtual IEnumerable<string> GetMessageIdsSince(DateTime dt, List<string> groupPatterns) {
			return new string[] { };
		}

		protected virtual Newsgroup GetGroup(string name) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected virtual NewsArticle GetArticle(string msgid) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected virtual NewsArticle GetArticle(Newsgroup g, int id) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected virtual IEnumerable<NewsArticle> GetArticles(Newsgroup g, int first, int? last) {
			foreach (int id in g.ArticleIds)
				if (id>=first && (!last.HasValue || id<=last))
					yield return GetArticle(g, id);
		}

		protected virtual void Post(MailMessageEx msg) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected virtual NewsUser OnAuth(string login, string password) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		protected override void OnRun() {
			WriteLine("200 {0}", Promt);
		}

		void CheckAuth() {
			if (User == null && AuthRequired)
				throw new NntpException(NntpStatusCode.AuthRequired);
		}

		void CheckSelectedGroup() {
			if (SelectedGroup == null)
				throw new NntpException(NntpStatusCode.NoGroupSelected);
		}

		void CheckSelectedArticle() {
			if (SelectedArticle == null)
				throw new NntpException(NntpStatusCode.NoArticleSelected);
		}

		static Regex s_reARTICLE = new Regex(@"<([^>]+)>|(\d+)?", RegexOptions.Compiled);

		NewsArticle FindArticle(string args, out bool byMsgId) {
			byMsgId = false;
			Match m = s_reARTICLE.Match(args);
			if (!m.Success)
				throw new SmtpException(SmtpStatusCode.SyntaxError);
			NewsArticle a;
			if (m.Groups[1].Value != "") {
				byMsgId = true;
				a = GetArticle(m.Groups[1].Value);
			}
			else if (m.Groups[2].Value != "") {
				CheckSelectedGroup();
				int id = Convert.ToInt32(m.Groups[2].Value);


				if (SelectedGroup.ArticleIds.Find(x => id==x) == -1)
					throw new NntpException(NntpStatusCode.NoSuchArticle);
				a = GetArticle(SelectedGroup, id);
			}
			else {
				CheckSelectedArticle();
				a = SelectedArticle;
			}
			return a;
		}

		protected virtual void OnARTICLE(string cmd, string args) {
			bool byMsgId;
			NewsArticle a = FindArticle(args, out byMsgId);
			Writer.WriteLine("220 {0} <{1}> article retrieved - head and body follow", a.Id, a.MessageId);
			Ut.WriteMailHeader(a.Headers, Writer);
			Writer.WriteLine();
			WriteLines(a.Body.Split(new char[]{'\n'}));
		}

		protected virtual void OnHEAD(string cmd, string args) {
			bool byMsgId;
			NewsArticle a = FindArticle(args, out byMsgId);
			Writer.WriteLine("221 {0} <{1}> article retrieved - head follows", a.Id, a.MessageId);
			Ut.WriteMailHeader(a.Headers, Writer);
			EndMultiline();
		}

		protected virtual void OnBODY(string cmd, string args) {
			bool byMsgId;
			NewsArticle a = FindArticle(args, out byMsgId);
			Writer.WriteLine("222 {0} <{1}> article retrieved - body follows", a.Id, a.MessageId);
			WriteLines(a.Body.Split(new char[]{'\n'}));
		}

		protected virtual void OnSTAT(string cmd, string args) {
			bool byMsgId;
			NewsArticle a = FindArticle(args, out byMsgId);
			WriteLine("223 {0} <{1}> article retrieved - request text separately", a.Id, a.MessageId);
			if (!byMsgId)
				SelectedArticle = a;
		}

		protected virtual void OnAUTHINFO(string cmd, string args) {
			Regex reAuthinfo = new Regex(@"(USER|PASS)\s+(\w+)", RegexOptions.IgnoreCase);
			Match m = reAuthinfo.Match(args);
			if (!m.Success)
				throw new SmtpException(SmtpStatusCode.SyntaxError);
			switch (m.Groups[1].Value.ToUpper()) {
				case "USER":
					Login = m.Groups[2].Value;
					WriteLine("381 More authentication information required");
					return;
				case "PASS":
					Password = m.Groups[2].Value;
					break;
			default:
					throw new SmtpException(SmtpStatusCode.SyntaxError);
			}
			User = OnAuth(Login, Password);
			ReadOnly = false;
			WriteLine("281 Authentication accepted");
		}

		protected virtual void OnDATE(string cmd, string args) {
			WriteLine("111 {0}", DateTime.Now.ToString("YYYYMMDDhhmmss"));
		}

		protected virtual void OnGROUP(string cmd, string args) {
			CheckAuth();
			string ggg = args.ToUpper();
			foreach (var s in GetLIST())
				if (s.ToUpper() == ggg) {
					SelectedGroup = GetGroup(s);
					WriteLine("201 {0} {1} {2} {3} group selected",
						SelectedGroup.EstimatedCount, SelectedGroup.First, SelectedGroup.Last, SelectedGroup.Name);
					List<int> aids = SelectedGroup.ArticleIds;
					if (aids.Count != 0)
						SelectedArticle = GetArticle(SelectedGroup, aids[0]);
					else
						SelectedArticle = null;
					return;
				}
			throw new NntpException(NntpStatusCode.NoSuchGroup);
		}

		protected virtual void OnHELP(string cmd, string args) {
			Writer.WriteLine("100 help text follows");
			WriteLines(CommandSet.Keys);
		}

		protected virtual void OnIHAVE(string cmd, string args) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		void SelectArticle(int id) {
			SelectedArticle = GetArticle(SelectedGroup, id);
			WriteLine("223 {0} {1} article retrieved", SelectedArticle.Id, SelectedArticle.MessageId);
		}

		protected virtual void OnLAST(string cmd, string args) {
			CheckSelectedGroup();
			CheckSelectedGroup();
			List<int> aids = SelectedGroup.ArticleIds;
			if (SelectedArticle.Id == aids[0])
				throw new NntpException(NntpStatusCode.NoPrevArticle);
			SelectArticle(aids[aids.Find(x => x==SelectedArticle.Id) - 1]);
		}

		protected virtual void OnNEXT(string cmd, string args) {
			CheckSelectedGroup();
			CheckSelectedGroup();
			List<int> aids = SelectedGroup.ArticleIds;
			if (SelectedArticle.Id == aids[aids.Count-1])
				throw new NntpException(NntpStatusCode.NoNextArticle);
			SelectArticle(aids[aids.Find(x => x == SelectedArticle.Id) + 1]);
		}

		protected virtual void OnLIST(string cmd, string args) {
			CheckAuth();
			if (args.ToUpper() == "OVERVIEW.FMT") {
				Writer.WriteLine("215 information follows");
				WriteLines(new string[] { "From:", "Date:", "Message-ID:", "References:", "Bytes:", "Lines:" });
			}
			else {
				Writer.WriteLine("215 list of newsgroups follows");
				WriteLines(GetLIST());
			}
		}

		protected virtual void OnNEWGROUPS(string cmd, string args) {
			Regex re = new Regex(@"(\d\d\d\d\d\d) (GMT)?");
			Match m = re.Match(args);
			if (!m.Success)
				throw new SmtpException(SmtpStatusCode.SyntaxError);
			DateTime dt = DateTime.ParseExact(m.Groups[1].Value, "YYMMDD", null);
			Writer.WriteLine("231 list of new newsgroups follows");
			WriteLines(GetGroupsSince(dt));
		}

		protected virtual void OnNEWNEWS(string cmd, string args) {
			Regex re = new Regex(@"(\S+)\s+(\d\d\d\d\d\d) (GMT)?");
			Match m = re.Match(args);
			if (!m.Success)
				throw new SmtpException(SmtpStatusCode.SyntaxError);
			DateTime dt = DateTime.ParseExact(m.Groups[2].Value, "YYMMDD", null);
			string groups = m.Groups[1].Value;
			Regex reGroup = new Regex(@"[^,]+");
			List<string> groupPatterns = new List<string>();
			foreach (Match ma in reGroup.Matches(groups))
				groupPatterns.Add(ma.Groups[0].Value);
			Writer.WriteLine("230 list of new articles by message-id follows");
			WriteLines(GetMessageIdsSince(dt, groupPatterns));
		}

		protected virtual void OnMODE(string cmd, string args) {
			if (ReadOnly)
				WriteLine("201 Hello, you can't post");
			else
				WriteLine("200 Hello, you can post");     
		}

		protected virtual void OnPOST(string cmd, string args) {
			if (ReadOnly)
				throw new NntpException(NntpStatusCode.PostingNotAllowed);
			WriteLine("340 send article to be posted. End with <CR-LF>.<CR-LF>");
			try {
				Post(ReadMessage());
			}
			catch (Exception x) {
				Console.Error.WriteLine(x.Message);
				Console.Error.WriteLine(x.StackTrace);
				throw new NntpException(NntpStatusCode.PostingFailed);
			}
			WriteLine("240 article posted ok");
		}

		protected virtual void OnQUIT(string cmd, string args) {
			WriteLine("205 closing connection - goodbye!");
			Exit = true;
		}

		protected virtual void OnSLAVE(string cmd, string args) {
			WriteLine("202 slave status noted");
		}

		protected virtual void OnXOVER(string cmd, string args) {
			Regex re = new Regex(@"(((\d+)(-?))(\d+)?)?");
			Match m = re.Match(args);
			if (!m.Success)
				throw new SmtpException(SmtpStatusCode.SyntaxError);
			Nullable<int> first = new Nullable<int>(),
						 last = new Nullable<int>();
			if (m.Groups[3].Value != "")
				first = Convert.ToInt32(m.Groups[3].Value);
			bool bMultiple = m.Groups[4].Value != "";
			if (m.Groups[5].Value != "")
				last = Convert.ToInt32(m.Groups[5].Value);

			CheckSelectedGroup();
			Writer.WriteLine("224 Overview information follows");
			if (bMultiple)
				foreach (var a in GetArticles(SelectedGroup, first.Value, last))
					Writer.WriteLine(a.GetOverview());
			else {
				CheckSelectedArticle();
				Writer.WriteLine(SelectedArticle.GetOverview());
			}
			EndMultiline();
		}

		protected virtual void NotImplemented(string cmd, string args) {
			throw new SmtpException(SmtpStatusCode.CommandNotImplemented);
		}

		public NntpServer() {
			CommandSet["ARTICLE"] = OnARTICLE;
			CommandSet["AUTHINFO"] = OnAUTHINFO;
			CommandSet["BODY"] = OnBODY;
			CommandSet["DATE"] = OnDATE;
			CommandSet["GROUP"] = OnGROUP;
			CommandSet["HEAD"] = OnHEAD;
			CommandSet["HELP"] = OnHELP;
			CommandSet["IHAVE"] = OnIHAVE;
			CommandSet["LAST"] = OnLAST;
			CommandSet["LIST"] = OnLIST;
			CommandSet["MODE"] = OnMODE;
			CommandSet["NEWGROUPS"] = OnNEWGROUPS;
			CommandSet["NEXT"] = OnNEXT;
			CommandSet["POST"] = OnPOST;
			CommandSet["QUIT"] = OnQUIT;
			CommandSet["SLAVE"] = OnSLAVE;
			CommandSet["STAT"] = OnSTAT;
			CommandSet["XOVER"] = OnXOVER;
		}
	}
}

