using System;
using System.IO;
using System.Security.Cryptography;

using Utils;

namespace Crypto
{
	public enum Scheme
	{
		RSA,
		Rijndael,
		TripleDES
	};

	class AsymEncAlgo
	{
		Scheme Scheme;
		AsymmetricAlgorithm Algo;

		public static AsymEncAlgo FromPubKey(byte[] pubKey)
		{
			AsymEncAlgo algo = new AsymEncAlgo();
			BinaryReader r = new BinaryReader(new MemoryStream(pubKey));
			switch (algo.Scheme = (Scheme)r.ReadByte())
			{
				case Scheme.RSA:
					string xmlKey = r.ReadString();
					RSACryptoServiceProvider rsa = new RSACryptoServiceProvider();
					rsa.FromXmlString(xmlKey);
					algo.Algo = rsa;
					break;
				default:
					throw new CryptographicException();
			}
			return algo;
		}

		public static AsymEncAlgo FromPrivKey(byte[] privKey)
		{
			return FromPubKey(privKey);
		}

		public byte[] Encrypt(byte[] data)
		{
			switch (Scheme)
			{
				case Scheme.RSA:
					RSACryptoServiceProvider rsa = Algo as RSACryptoServiceProvider;
					MemoryStream ms = new MemoryStream();
					using (BinaryWriter w = new BinaryWriter(ms))
					{
						Scheme sch = Scheme.TripleDES;
						w.Write((byte)sch);
						SymmetricAlgorithm symAlgo;
						switch (sch)
						{
							case Scheme.Rijndael: //!!! this not supported everywhere
								symAlgo = new RijndaelManaged();
								break;
							case Scheme.TripleDES:
								symAlgo = new TripleDESCryptoServiceProvider();
								break;
							default:
								throw new CryptographicException();
						}
						Ut.Write(w,rsa.Encrypt(symAlgo.Key,false));
						Ut.Write(w,rsa.Encrypt(symAlgo.IV,false));
						using (BinaryWriter cw = new BinaryWriter(new CryptoStream(ms,symAlgo.CreateEncryptor(),CryptoStreamMode.Write)))
							Ut.Write(cw,data);
					}
					return ms.ToArray();
				default:
					throw new CryptographicException();
			}
		}

		public byte[] Decrypt(byte[] data)
		{
			switch (Scheme)
			{
				case Scheme.RSA:
					RSACryptoServiceProvider rsa = Algo as RSACryptoServiceProvider;
					MemoryStream ms = new MemoryStream(data);
					SymmetricAlgorithm symAlgo;
					using (BinaryReader r = new BinaryReader(ms))
					{
						switch ((Scheme)r.ReadByte())
						{
							case Scheme.Rijndael:
								symAlgo = new RijndaelManaged();
								break;
							case Scheme.TripleDES:
								symAlgo = new TripleDESCryptoServiceProvider();
								break;
							default:
								throw new CryptographicException();
						}
						symAlgo.Key = rsa.Decrypt(Ut.ReadByteArray(r),false);
						symAlgo.IV = rsa.Decrypt(Ut.ReadByteArray(r),false);
						using (BinaryReader cr = new BinaryReader(new CryptoStream(ms,symAlgo.CreateDecryptor(),CryptoStreamMode.Read)))
							return Ut.ReadByteArray(cr);
					}
				default:
					throw new CryptographicException();
			}
		}
	}

	public class CryptoKeys
	{
		public Scheme Scheme = Scheme.RSA;
		public byte[] PrivateKey,
			            PublicKey;
	}

	class Cryptor
	{
		static AsymmetricSignatureFormatter CreateSignatureFormatter(byte[] privKey)
		{
			AsymmetricSignatureFormatter f;
			BinaryReader r = new BinaryReader(new MemoryStream(privKey));
			Scheme scheme = (Scheme)r.ReadByte();
			switch (scheme)
			{
				case Scheme.RSA:
					string xmlKey = r.ReadString();
					RSACryptoServiceProvider rsa = new RSACryptoServiceProvider();
					rsa.FromXmlString(xmlKey);
					f = new RSAPKCS1SignatureFormatter(rsa);
					break;
				default:
					throw new CryptographicException();
			}
			f.SetHashAlgorithm("SHA1");
			return f;
		}

		static AsymmetricSignatureDeformatter CreateSignatureDeformatter(byte[] pubKey)
		{
			AsymmetricSignatureDeformatter df;
			BinaryReader r = new BinaryReader(new MemoryStream(pubKey));
			Scheme scheme = (Scheme)r.ReadByte();
			switch (scheme)
			{
				case Scheme.RSA:
					string xmlKey = r.ReadString();
					RSACryptoServiceProvider rsa = new RSACryptoServiceProvider();
					rsa.FromXmlString(xmlKey);
					df = new RSAPKCS1SignatureDeformatter(rsa);
					break;
				default:
					throw new CryptographicException();
			}
			df.SetHashAlgorithm("SHA1");
			return df;
		}

		public static byte[] Encrypt(byte[] pubKey, byte[] data)
		{
			return AsymEncAlgo.FromPubKey(pubKey).Encrypt(data);
		}

		public static byte[] Decrypt(byte[] privKey, byte[] data)
		{
			return AsymEncAlgo.FromPrivKey(privKey).Decrypt(data);
		}

		public static byte[] Sign(byte[] privKey, byte[] data)
		{
			SHA1 hash = new SHA1Managed();
			hash.TransformFinalBlock(data,0,data.Length);
			return CreateSignatureFormatter(privKey).CreateSignature(hash);
		}

		public static bool Verify(byte[] pubKey, byte[] data, byte[] signature)
		{
			SHA1 hash = new SHA1Managed();
			hash.TransformFinalBlock(data,0,data.Length);
			return CreateSignatureDeformatter(pubKey).VerifySignature(hash,signature);
		}

		public static CryptoKeys GenerateKeys(Scheme scheme)
		{
			CryptoKeys keys = new CryptoKeys();
			switch (scheme)
			{
				case Scheme.RSA:
					RSACryptoServiceProvider rsa = new RSACryptoServiceProvider();
					using (MemoryStream ms = new MemoryStream())
						using (BinaryWriter w = new BinaryWriter(ms))
						{
							w.Write((byte)scheme);
							w.Write(rsa.ToXmlString(true));
							keys.PrivateKey = ms.ToArray();
						}
					using (MemoryStream ms = new MemoryStream())
						using (BinaryWriter w = new BinaryWriter(ms))
						{
							w.Write((byte)scheme);
							w.Write(rsa.ToXmlString(false));
							keys.PublicKey = ms.ToArray();
						}
					break;
				default:
					throw new CryptographicException();
			}
			return keys;
		}

		public static CryptoKeys GenerateKeys()
		{
			return GenerateKeys(Scheme.RSA);
		}
	};

}



