using System;
using System.IO;
using System.Net;
using System.Runtime.Serialization.Formatters.Binary;
using System.Collections;
using System.Windows.Forms;
using System.Diagnostics;
using Microsoft.Win32;
using System.Threading;
using System.Security.Cryptography;

using Crypto;

using Utils;
using Unet;

namespace Mess
{
	public enum AccountState : byte
	{
		Offline,
		Online,
		Connecting,
		Away,
		NA,
		Occupied,
		FreeForChat,
		DND,
		Invisible
	}

	[Serializable]
	public class AccountInfo : IWritable
	{
		public enum GenderEnum : byte
		{
			Undefined,
			Male,
			Female
		}

		public Guid Guid = Guid.NewGuid();
		public UInt32 Uin;
		public DateTime UpdateTime;
		public string NickName,
					  FirstName,
					  LastName,
					  Email,
					  Homepage,
					  Country,
					  State,
					  City,
					  UserInfo = ""; //!!!
		public DateTime Birthday;
		public GenderEnum Gender;
		public byte[] PublicKey;
		internal byte[] Data,
						Signature;

		[NonSerialized]
		public AccountState AccountState;

		[NonSerialized]
		internal DateTime TimeLastReceived;
		[NonSerialized]
		internal bool Pinged;

		internal byte[] CreateData()
		{
			UpdateTime = DateTime.Now;
			return Data = Ut.ToArray(new AccountInfoData(this));
		}

		public override void Read(BinaryReader r)
		{
			Guid = Ut.ReadGuid(r);
			Uin = r.ReadUInt32();
			UpdateTime = Ut.ReadDateTime(r);
			NickName = r.ReadString();
			FirstName = r.ReadString();
			LastName = r.ReadString();
			Email = r.ReadString();
			Homepage = r.ReadString();
			Country = r.ReadString();
			State = r.ReadString();
			City = r.ReadString();
			UserInfo = r.ReadString();
			Birthday = Ut.ReadDateTime(r);
			Gender = (GenderEnum)r.ReadByte();
			PublicKey = Ut.ReadByteArray(r);
			Data = Ut.ReadByteArray(r);
			Signature = Ut.ReadByteArray(r);
		}

		public override void Write(BinaryWriter w)
		{
			Ut.Write(w, Guid);
			w.Write(Uin);
			Ut.Write(w, UpdateTime);
			w.Write(NickName);
			w.Write(FirstName);
			w.Write(LastName);
			w.Write(Email);
			w.Write(Homepage);
			w.Write(Country);
			w.Write(State);
			w.Write(City);
			w.Write(UserInfo);
			Ut.Write(w, Birthday);
			w.Write((byte)Gender);
			Ut.Write(w, PublicKey);
			Ut.Write(w, Data);
			Ut.Write(w, Signature);
		}

		public void Sign()
		{
			Data = null;
			Signature = null;
			Signature = Cryptor.Sign(Messenger.Instance.AccountMan.Profile.PrivateKey, CreateData());
		}

		public void CopyFrom(AccountInfo ai)
		{
			lock (Messenger.Instance)
			{
				Guid = ai.Guid;
				Uin = ai.Uin;
				UpdateTime = ai.UpdateTime;
				NickName = ai.NickName;
				FirstName = ai.FirstName;
				LastName = ai.LastName;
				Email = ai.Email;
				Country = ai.Country;
				City = ai.City;
				State = ai.State;
				Homepage = ai.Homepage;
				UserInfo = ai.UserInfo;
				Birthday = ai.Birthday;
				Gender = ai.Gender;
				PublicKey = ai.PublicKey;
				Data = ai.Data;
				Signature = ai.Signature;
			}
		}
	}

	public class NoAccountInfoException : Exception
	{
	}

	[Serializable]
	public abstract class HistoryItem
	{
		public DateTime DateTime;
		public bool ReceivedItem;
	}

	[Serializable]
	public class MessageHistoryItem : HistoryItem
	{
		public string Text;
	}

	[Serializable]
	public class ContactInfo
	{
		public AccountInfo AccountInfo;

		string nickName;

		public string NickName
		{
			get
			{
				string s = nickName != null ? nickName : AccountInfo.NickName;
				if (s == "")
					s = "Unknown";
				return s;
			}
			set
			{
				nickName = value;
			}
		}

		public ArrayList History = new ArrayList();
		public ArrayList UnreadMessages = new ArrayList();
		public bool Temporary;

		public ContactInfo(AccountInfo ai)
		{
			AccountInfo = ai;
		}
	}

	abstract class MessengerPacket : Unet.Packet
	{
		virtual internal void Process(Messenger mess)
		{
			DateTime dtForPing = DateTime.Now - TimeSpan.FromSeconds(Params.MessengerPingTimeOut),
					 dtForPing2 = DateTime.Now - TimeSpan.FromSeconds(Params.MessengerPingTimeOut * 2);
			try
			{
				lock (mess)
					foreach (DictionaryEntry de in mess.AccountMan.HtAccount)
					{
						AccountInfo ai = ((DataForAccount)de.Value).AccountInfo;
						if (mess.AccountMan.Profile != null && mess.AccountMan.Profile.AccountInfo == ai)
							continue;
						if (ai.AccountState == AccountState.Online)
							if (ai.Pinged)
							{
								if (ai.TimeLastReceived < dtForPing2)
								{
									ai.Pinged = false;
									ai.AccountState = AccountState.Offline;
									mess.StateChanged(ai);
								}
							}
							else if (ai.TimeLastReceived < dtForPing)
							{
								mess.UnetMan.Relay(new QueryAccountStatePacket(ai));
								ai.Pinged = true;
							}
					}
			}
			catch (NoAccountInfoException)
			{ }
		}

		public override void Process()
		{
			Process(Messenger.Instance);
		}
	}

	[Serializable]
	public class DataForAccount
	{
		internal AccountInfo AccountInfo;
		internal Hashtable HtData = new Hashtable(new BlobComparer());

		internal DataForAccount(AccountInfo ai)
		{
			AccountInfo = ai;
		}
	}

	class SignedPacket : MessengerPacket
	{
		internal byte[] Data;
		internal Guid GuidFrom;
		internal byte[] Signature;

		public SignedPacket()
		{ }

		void Sign(FromData fromData)
		{
			Profile prof = Messenger.Instance.AccountMan.Profile;
			Signature = Cryptor.Sign(prof.PrivateKey, Data = Ut.ToArray(fromData));
		}

		internal SignedPacket(FromData fromData)
		{
			Logger.Log("SignedPacket constructor");
			Profile prof = Messenger.Instance.AccountMan.Profile;
			GuidFrom = prof.AccountInfo.Guid;
			Logger.Log("Before Sign");
			Sign(fromData);
			Logger.Log("After Sign");
		}

		internal SignedPacket(AccountInfo ai)
		{
			Data = ai.Data;
			GuidFrom = ai.Guid;
			Signature = ai.Signature;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Data = Ut.ReadByteArray(r);
			GuidFrom = Ut.ReadGuid(r);
			Signature = Ut.ReadByteArray(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(Messenger.PACKET_SIGNED);
			Ut.Write(w, Data);
			Ut.Write(w, GuidFrom);
			Ut.Write(w, Signature);
		}

		FromData GetFromData()
		{
			BinaryReader r = new BinaryReader(new MemoryStream(Data));
			byte typ = r.ReadByte();
			FromData fd;
			switch (typ)
			{
				case FromData.TEXT_DATA: fd = new TextData(); break;
				case FromData.ACCOUNTINFO_DATA: fd = new AccountInfoData(); break;
				case FromData.STATE_DATA: fd = new StateData(); break;
				case FromData.CONFIRMATION_DATA: fd = new ConfirmationData(); break;
				default: return null;
			}
			fd.Read(r);
			return fd;

			//!!!return (FromData)UnetUtils.FromData(Data);
		}

		internal override void Process(Messenger mess)
		{
			FromData fromData = GetFromData();
			Logger.Log("Received: {0}", fromData);
			if (!(fromData is AccountInfoData))
			{
				DataForAccount dfa;
				lock (mess)
					dfa = (DataForAccount)mess.AccountMan.HtAccount[GuidFrom];
				if (dfa == null)
				{
					mess.UnetMan.Relay(new QueryAccountInfoPacket(GuidFrom));
					throw new NoAccountInfoException();
				}
				else
				{
					AccountInfo ai = dfa.AccountInfo;
					ai.TimeLastReceived = DateTime.Now;
					if (!Cryptor.Verify(ai.PublicKey, Data, Signature))
						throw new Exception("Invalid signature");
					lock (mess)
						fromData.Process(dfa);
				}
			}
			lock (mess)
				fromData.Process(this);
			base.Process(mess);
		}
	}

	[Serializable]
	abstract class FromData : IWritable
	{
		public const byte TEXT_DATA = 0,
						  ACCOUNTINFO_DATA = 1,
						  STATE_DATA = 2,
						  CONFIRMATION_DATA = 3;

		internal virtual void Process(SignedPacket signedPacket)
		{
		}

		internal virtual void Process(DataForAccount dfa)
		{
		}
	}

	[Serializable]
	class TextData : FromData
	{
		internal DateTime DateTime;
		internal string Text;
		internal Guid GuidTo;

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			DateTime = Ut.ReadDateTime(r);
			Text = r.ReadString();
			GuidTo = Ut.ReadGuid(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(TEXT_DATA);
			Ut.Write(w, DateTime);
			w.Write(Text);
			Ut.Write(w, GuidTo);
		}

		internal override void Process(SignedPacket signedPacket)
		{
			Messenger mess = Messenger.Instance;
			if (GuidTo != mess.AccountMan.Profile.AccountInfo.Guid)
				throw new Exception("Faked GuidTo");
			ContactInfo contact = null;
			ArrayList cl = mess.AccountMan.Profile.ContactList;
			lock (mess)
			{
				foreach (ContactInfo ci in cl)
					if (ci.AccountInfo.Guid == signedPacket.GuidFrom)
					{
						contact = ci;
						break;
					}
				if (contact == null)
				{

					DataForAccount dfa = (DataForAccount)mess.AccountMan.HtAccount[signedPacket.GuidFrom];
					AccountInfo ai = dfa.AccountInfo;
					cl.Add(contact = new ContactInfo(ai));
					contact.Temporary = true;
					mess.UpdateView();
				}
			}
			MessageHistoryItem mhi = new MessageHistoryItem();
			mhi.Text = Text;
			mhi.DateTime = DateTime;
			mhi.ReceivedItem = true;
			contact.UnreadMessages.Add(mhi);
			mess.NotifyAboutMessage();
		}
	}

	[Serializable]
	class AccountInfoData : FromData
	{
		AccountInfo AccountInfo;

		public AccountInfoData()
		{
		}

		internal AccountInfoData(AccountInfo ai)
		{
			AccountInfo = ai;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			AccountInfo = new AccountInfo();
			AccountInfo.Read(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(ACCOUNTINFO_DATA);
			AccountInfo.Write(w);
		}

		internal override void Process(SignedPacket signedPacket)
		{
			if (!signedPacket.GuidFrom.Equals(AccountInfo.Guid))
				throw new Exception("Faked AccountInfo");
			AccountInfo.Data = signedPacket.Data;
			AccountInfo.Signature = signedPacket.Signature;
			Messenger mess = Messenger.Instance;
			AccountInfo ai = null;
			lock (mess)
			{
				DataForAccount dfa = (DataForAccount)mess.AccountMan.HtAccount[AccountInfo.Guid];
				if (dfa != null)
				{
					if (AccountInfo.UpdateTime > dfa.AccountInfo.UpdateTime)
						(ai = dfa.AccountInfo).CopyFrom(AccountInfo);
				}
				else
				{
					mess.AccountMan.HtAccount[AccountInfo.Guid] = new DataForAccount(ai = AccountInfo);
					Profile prof = mess.AccountMan.Profile;
					if (prof != null)
						mess.UnetMan.Relay(new SignedPacket(new StateData(prof.AccountInfo.AccountState)));
					Logger.Log("New account: {0}", AccountInfo.NickName); //!!!
				}
			}
			if (ai != null)
				mess.AccountInfoReceived(ai);
		}
	}

	[Serializable]
	class StateData : FromData
	{
		internal AccountState AccountState;

		public StateData()
		{
		}

		internal StateData(AccountState state)
		{
			Logger.Log("StateData(AccountState state)");
			AccountState = state;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			AccountState = (AccountState)r.ReadByte();
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(STATE_DATA);
			w.Write((byte)AccountState);
		}

		internal override void Process(DataForAccount dfa)
		{
			Messenger mess = Messenger.Instance;
			dfa.AccountInfo.AccountState = AccountState;
			mess.StateChanged(dfa.AccountInfo);
			if (dfa.AccountInfo.AccountState == AccountState.Online)
				lock (mess)
					foreach (DictionaryEntry de in dfa.HtData)
						if (de.Value != null)
							mess.UnetMan.Relay(new EncryptedPacket(dfa.AccountInfo.Guid, (byte[])de.Value));
		}
	}

	[Serializable]
	class ConfirmationData : FromData
	{
		byte[] Hash;

		internal ConfirmationData()
		{
		}

		internal ConfirmationData(byte[] hash)
		{
			Hash = hash;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Hash = Ut.ReadByteArray(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(CONFIRMATION_DATA);
			Ut.Write(w, Hash);
		}

		internal override void Process(DataForAccount dfa)
		{
			dfa.HtData[Hash] = null;
		}
	}

	class EncryptedPacket : MessengerPacket
	{
		internal Guid GuidTo;
		internal byte[] Data;

		public EncryptedPacket()
		{
		}

		internal EncryptedPacket(Guid guid, byte[] data)
		{
			GuidTo = guid;
			Data = data;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Data = Ut.ReadByteArray(r);
			GuidTo = Ut.ReadGuid(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(Messenger.PACKET_ENCRYPTED);
			Ut.Write(w, Data);
			Ut.Write(w, GuidTo);
		}

		internal override void Process(Messenger mess)
		{
			if (mess.AccountMan.Profile != null && mess.AccountMan.Profile.AccountInfo.Guid == GuidTo)
			{
				Msg.DisableRelay = true;
				mess.AccountMan.Profile.Process(Data, mess);
			}
			else
				lock (mess)
				{
					DataForAccount dfa = (DataForAccount)mess.AccountMan.HtAccount[GuidTo];
					if (dfa != null)
					{
						SHA1 sha1 = new SHA1Managed();
						byte[] hash = sha1.TransformFinalBlock(Data, 0, Data.Length);
						if (dfa.HtData.Contains(hash))
						{
							if (dfa.HtData[hash] == null)
								Msg.DisableRelay = true;
						}
						else
							dfa.HtData[hash] = Data;
					}
				}
			base.Process(mess);
		}
	}

	class QueryAccountStatePacket : MessengerPacket
	{
		Guid Guid;

		public QueryAccountStatePacket()
		{
		}

		internal QueryAccountStatePacket(AccountInfo ai)
		{
			Guid = ai.Guid;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Guid = Ut.ReadGuid(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(Messenger.PACKET_QUERYACCOUNTSTATE);
			Ut.Write(w, Guid);
		}

		internal override void Process(Messenger mess)
		{
			Profile prof = mess.AccountMan.Profile;
			if (prof != null && prof.AccountInfo.Guid == Guid)
				mess.UnetMan.Relay(new SignedPacket(new StateData(prof.AccountInfo.AccountState)));
			base.Process(mess);
		}
	}

	class QueryAccountInfoPacket : MessengerPacket
	{
		internal Guid Guid;

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Guid = Ut.ReadGuid(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(Messenger.PACKET_QUERYACCOUNTINFO);
			Ut.Write(w, Guid);
		}

		internal QueryAccountInfoPacket()
		{ }

		internal QueryAccountInfoPacket(Guid guid)
		{
			Guid = guid;
		}

		internal override void Process(Messenger mess)
		{
			lock (mess)
			{
				if (Guid == Guid.Empty)
					foreach (DictionaryEntry de in mess.AccountMan.HtAccount)
						mess.UnetMan.Relay(new SignedPacket(((DataForAccount)de.Value).AccountInfo));
				else
				{
					DataForAccount dfa = (DataForAccount)mess.AccountMan.HtAccount[Guid];
					if (dfa != null)
					{
						AccountInfo ai = dfa.AccountInfo;
						if (ai.Data != null)
							mess.UnetMan.Relay(new SignedPacket(ai));
					}
				}
			}
			base.Process(mess);
		}
	}


	[Serializable]
	class BlobComparer : IEqualityComparer
	{
		bool IEqualityComparer.Equals(object x, object y)
		{
			byte[] a = (byte[])x,
				   b = (byte[])y;
			if (a.Length < b.Length)
				return false;
			else if (a.Length > b.Length)
				return false;
			for (int i = 0; i < a.Length; i++)
				if (a[i] != b[i])
					return a[i] == b[i];
			return true;
		}

		public int GetHashCode(object x)
		{
			int r = 0;
			byte[] a = (byte[])x;
			for (int i = 0; i < a.Length; i++)
				r += a[i];
			return r;
		}
	}

	[Serializable]
	public class Profile
	{
		public AccountInfo AccountInfo;
		internal Crypto.Scheme Scheme;
		internal byte[] PrivateKey;
		internal Hashtable ReceivedHashes = new Hashtable(new BlobComparer());
		internal int CurrentHashNumber;

		public ArrayList ContactList = new ArrayList();

		internal void Process(byte[] data, Messenger mess)
		{
			SHA1 sha1 = new SHA1Managed();
			byte[] hash = sha1.TransformFinalBlock(data, 0, data.Length);
			lock (mess)
				if (ReceivedHashes.Contains(hash))
					mess.UnetMan.Relay(new SignedPacket(new ConfirmationData(hash)));
				else
					try
					{
						MessengerPacket mp = (MessengerPacket)Packet.Deserialize(new BinaryReader(new MemoryStream(Cryptor.Decrypt(PrivateKey, data))));
						mp.Process(mess);

						//!!!            ((MessengerPacket)UnetUtils.FromData()).Process(mess);
						ReceivedHashes[hash] = ++CurrentHashNumber;
					}
					catch (NoAccountInfoException)
					{ }
		}
	}

	[Serializable]
	public class AccountMan
	{
		internal Hashtable HtAccount = new Hashtable();
		internal Profile Profile;
	}

	class AccountUpdateDateComparer : IComparer
	{
		public int Compare(object x, object y)
		{
			return DateTime.Compare(((AccountInfo)x).UpdateTime, ((AccountInfo)y).UpdateTime);
		}
	}

	public class Messenger : IDisposable
	{
		internal const byte
		  PACKET_SIGNED = 8,
		  PACKET_ENCRYPTED = 9,
		  PACKET_QUERYACCOUNTSTATE = 10,
		  PACKET_QUERYACCOUNTINFO = 11;

		internal AccountMan AccountMan = new AccountMan();

		public delegate void AccountInfoHandler(object o, AccountInfo ai);
		public event AccountInfoHandler OnAccountInfo;

		//!!!    public delegate void NotifyAboutMessageHandler();
		public event SimpleDelegateHandler OnNotifyAboutMessage;

		public delegate void ChangeContactStateHandler(object o, EventArgs e);
		public event ChangeContactStateHandler OnChangeContactState;

		public static Messenger Instance;
		public UnetMan UnetMan;

		string GetDataFileDir()
		{
			return Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData), @"Ufasoft\P2PMess\");
		}

		public Messenger(UnetMan unetMan)
		{
			(UnetMan = unetMan).Services.Add(Instance = this);
			BinaryFormatter f = new BinaryFormatter();
			FileStream stm = null;
			try
			{
				stm = new FileStream(GetDataFileDir() + "p2pmess.dat", FileMode.Open, FileAccess.Read);
				AccountMan = (AccountMan)f.Deserialize(stm);
				ArrayList ar = (ArrayList)f.Deserialize(stm);
				foreach (Peer p in ar)
					UnetMan.AddLiveEntry(p.ID, p.TcpEndPoint, p.UdpEndPoint);
			}
			catch (Exception)
			{
				if (stm != null)
					stm.Close();
				try
				{
					stm = new FileStream(GetDataFileDir() + "account.dat", FileMode.Open, FileAccess.Read);
					AccountMan.Profile = (Profile)f.Deserialize(stm);
					AccountMan.HtAccount[AccountMan.Profile.AccountInfo.Guid] = new DataForAccount(AccountMan.Profile.AccountInfo);
				}
				catch (Exception)
				{ }
			}
			finally
			{
				if (stm != null)
					stm.Close();
			}
			DeleteOldAccounts();
			if (AccountMan.Profile != null)
				AccountMan.Profile.AccountInfo.AccountState = AccountState.Connecting;
		}

		void DeleteOldAccounts()
		{
			ArrayList ar = new ArrayList();
			foreach (DictionaryEntry de in AccountMan.HtAccount)
				ar.Add(((DataForAccount)de.Value).AccountInfo);
			ar.Sort(new AccountUpdateDateComparer());
			if (AccountMan.Profile != null)
				for (int i = AccountMan.Profile.ContactList.Count - 1; i >= 0; i--)
				{
					ContactInfo ci = (ContactInfo)AccountMan.Profile.ContactList[i];
					if (ci.Temporary)
						AccountMan.Profile.ContactList.RemoveAt(i);
				}
			for (int i = 0; i < ar.Count - Params.MaxSavedAccounts; i++)
			{
				AccountInfo ai = (AccountInfo)ar[i];
				if (AccountMan.Profile != null)
				{
					if (AccountMan.Profile.AccountInfo == ai)
						goto found;
					foreach (ContactInfo ci in AccountMan.Profile.ContactList)
						if (ci.AccountInfo == ai)
							goto found;
				}
				AccountMan.HtAccount.Remove(ai.Guid);
			found:
				;
			}
		}

		public void Save()
		{
			string dir = GetDataFileDir();
			Directory.CreateDirectory(dir);
			BinaryFormatter f = new BinaryFormatter();
			using (FileStream stm = new FileStream(dir + "p2pmess.dat", FileMode.Create, FileAccess.Write))
			{
				lock (this)
					f.Serialize(stm, AccountMan);
				ArrayList ar = new ArrayList();
				lock (UnetMan.Instance.PeerMan)
					foreach (Peer peer in UnetMan.Instance.PeerMan.ActivePeers)
						if (!peer.Bad)
							ar.Add(peer);
				f.Serialize(stm, ar);
			}
			Profile prof = new Profile();
			using (FileStream stm = new FileStream(GetDataFileDir() + "account.dat", FileMode.Create, FileAccess.Write))
				lock (this)
				{
					prof.AccountInfo = AccountMan.Profile.AccountInfo;
					prof.PrivateKey = AccountMan.Profile.PrivateKey;
					f.Serialize(stm, prof);
				}
		}

		internal void NotifyAboutMessage()
		{
			if (OnNotifyAboutMessage != null)
				OnNotifyAboutMessage();
		}

		public void AccountInfoReceived(AccountInfo ai)
		{
			if (OnAccountInfo != null)
				OnAccountInfo(this, ai);
		}

		public void InitialOnAccountInfo()
		{
			foreach (DictionaryEntry de in AccountMan.HtAccount)
				AccountInfoReceived(((DataForAccount)de.Value).AccountInfo);
		}

		void OnConnectionStateChanged(bool connected)
		{
			//!!!Thread.Sleep(3000);//!!!T
			lock (this)
			{
				Profile prof = AccountMan.Profile;
				if (prof != null)
				{
					AccountInfo ai = prof.AccountInfo;
					if (connected)
					{
						if (ai.AccountState == AccountState.Connecting)
							ai.AccountState = AccountState.Online;
						Logger.Log("Connected in OnConnectionStateChanged");
						SetState(ai.AccountState);
					}
					else
					{
						if (ai.AccountState != AccountState.Offline)
							ai.AccountState = AccountState.Connecting;
					}
					foreach (ContactInfo ci in AccountMan.Profile.ContactList)
						UnetMan.Relay(new QueryAccountStatePacket(ci.AccountInfo));
					UpdateView();
				}
			}
		}

		internal void UpdateView()
		{
			if (OnChangeContactState != null)
				OnChangeContactState(null, null);
		}

		void OnDeserializePacket(byte typ, BinaryReader r, ref Packet pack)
		{
			switch (typ)
			{
				case PACKET_SIGNED: pack = new SignedPacket(); break;
				case PACKET_ENCRYPTED: pack = new EncryptedPacket(); break;
				case PACKET_QUERYACCOUNTSTATE: pack = new QueryAccountStatePacket(); break;
				case PACKET_QUERYACCOUNTINFO: pack = new QueryAccountInfoPacket(); break;
			}
		}

		public void Create()
		{
			UnetMan.OnConnectionStateChanged += new UnetMan.ConnectionStateChangedHandler(OnConnectionStateChanged);
			Packet.DeserializePacket += new Packet.DeserializePacketHandler(OnDeserializePacket);
			OnConnectionStateChanged(UnetMan.Connected);
			UpdateView();
		}

		void IDisposable.Dispose()
		{
			Packet.DeserializePacket -= new Packet.DeserializePacketHandler(OnDeserializePacket);
			UnetMan.OnConnectionStateChanged -= new UnetMan.ConnectionStateChangedHandler(OnConnectionStateChanged);
		}

		public void CreateAccount(AccountInfo ai)
		{
			CryptoKeys keys = Cryptor.GenerateKeys();
			Profile prof = AccountMan.Profile = new Profile();
			prof.AccountInfo = ai;
			prof.Scheme = keys.Scheme;
			prof.PrivateKey = keys.PrivateKey;
			ai.PublicKey = keys.PublicKey;
			ai.Sign();
			ai.AccountState = AccountState.Connecting;
			lock (this)
				AccountMan.HtAccount[ai.Guid] = new DataForAccount(ai);
			AfterUpdateAccount();
			OnConnectionStateChanged(UnetMan.Connected);
		}

		public void AfterUpdateAccount()
		{
			Save();
			UpdateView();
			UnetMan.Relay(new SignedPacket(AccountMan.Profile.AccountInfo));
		}

		public void AddToProfile(AccountInfo ai)
		{
			if (AccountMan.Profile.AccountInfo == ai)
				return;
			foreach (ContactInfo ci in AccountMan.Profile.ContactList)
				if (ci.AccountInfo == ai)
					return;
			AccountMan.Profile.ContactList.Add(new ContactInfo(ai));
		}

		public void DeleteFromProfile(ContactInfo ci)
		{
			lock (this)
				AccountMan.Profile.ContactList.Remove(ci);
		}

		public void SetState(AccountState state)
		{
			Logger.Log("Messenger setting state");
			if (AccountMan == null)
				Logger.Log("AccountMan == null!!!");
			if (AccountMan.Profile != null)
				UnetMan.Relay(new SignedPacket(new StateData(state)));
			Logger.Log("Messenger setted state");
		}

		internal void SendTo(AccountInfo ai, SignedPacket m)
		{
			EncryptedPacket em = new EncryptedPacket(ai.Guid, Cryptor.Encrypt(ai.PublicKey, Ut.ToArray(m)));
			UnetMan.Relay(em);
			em.Process();
		}

		public void SendMessage(ContactInfo ci, string s)
		{
			TextData td = new TextData();
			td.GuidTo = ci.AccountInfo.Guid;
			td.DateTime = DateTime.Now;
			td.Text = s;
			SendTo(ci.AccountInfo, new SignedPacket(td));
			MessageHistoryItem mhi = new MessageHistoryItem();
			mhi.Text = s;
			mhi.DateTime = td.DateTime;
			lock (this)
				ci.History.Add(mhi);
		}

		public void StateChanged(AccountInfo ai)
		{
			if (OnChangeContactState != null)
				OnChangeContactState(this, null);
		}
	}
}
