using System;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.Collections;
using System.Threading;
using System.IO;
using System.Text.RegularExpressions;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Diagnostics;
using System.Reflection;

using Utils;
using ProxyUtils;


namespace Unet
{
	


	public class Msg
	{
		public Byte TTL = Params.DefaultTTL,
			Hops,
			Ver = 2;
		public Guid Guid = Guid.NewGuid();
		public byte[] Data;
		public Peer PeerFrom;
		public IPEndPoint UdpEndPoint;
		public bool DisableRelay;

		public void Read(BinaryReader r)
		{
			Read(r,r.ReadByte());
		}

		public void Read(BinaryReader r, byte ttl)
		{
			TTL = ttl;
			Hops = r.ReadByte();
			Ver = r.ReadByte();
			Guid = Ut.ReadGuid(r);
			Int32 len;
			len = r.ReadInt16();
			Data = r.ReadBytes(len);
			//!!!      Data = new Byte[len];
			//!!!      r.Read(Data,0,len);
		}

		public void Write(BinaryWriter w)
		{
			w.Write(TTL);
			w.Write(Hops);
			w.Write(Ver);
			Ut.Write(w,Guid);
			if (Data == null)
				w.Write((Int16)0);
			else
			{
				w.Write((Int16)Data.Length);
				w.Write(Data);
			}
			w.Flush();
		}

		public Packet ToPacket()
		{
			//Logger.Log("Packet from "+PeerFrom.ToEndPointString());
			return Packet.Deserialize(new BinaryReader(new MemoryStream(Data)));
		}

		[Conditional("DEBUG")]
		internal void Log()
		{
			Logger.Log("{0} TTL={1}",this,TTL);//!!!
		}    
	}

	abstract public class Packet : IWritable
	{
		protected const byte PACKET_TIME          = 0,
			PACKET_QUERYTIME     = 1,
			PACKET_VERSION       = 2,
			PACKET_QUERYVERSION  = 3,
			PACKET_PEER          = 4,
			PACKET_QUERYPEERS    = 5,
			PACKET_PING          = 6,

			PACKET_QUERYSTAT     = 20,
			PACKET_STAT          = 21;


		[NonSerialized] public Msg Msg = new Msg();

		[Conditional("DEBUG")]
		internal void Log()
		{
			if (Msg.PeerFrom != null)
				Logger.Log("{0} TTL={1} From {2}",this,Msg.TTL,Msg.PeerFrom);//!!!
		}

		public Packet()
		{
			Logger.Log("Packet()");
		}

		public virtual void Process()
		{
		}

		public virtual void BeforeSend()
		{
		}

		public delegate void DeserializePacketHandler(byte typ, BinaryReader r, ref Packet pack);
		static public event DeserializePacketHandler DeserializePacket;

		public static Packet Deserialize(BinaryReader r)
		{
			byte typ = r.ReadByte();
			Packet pack = null;
			switch (typ)
			{
				case PACKET_TIME:         pack = new TimePacket(); break;
				case PACKET_QUERYTIME:    pack = new QueryTimePacket(); break;
				case PACKET_VERSION:      pack = new VersionPacket(); break;
				case PACKET_QUERYVERSION: pack = new QueryVersionPacket(); break;
				case PACKET_PEER:         pack = new PeerPacket(); break;
				case PACKET_QUERYPEERS:   pack = new QueryPeersPacket(); break;
				case PACKET_PING:         pack = new PingPacket(); break;
				case PACKET_STAT:         pack = new StatPacket(); break;
				case PACKET_QUERYSTAT:    pack = new QueryStatPacket(); break;
				default:
					if (DeserializePacket != null)
						DeserializePacket(typ,r,ref pack);
					break;
			}
			if (pack == null)
				return null;
			pack.Read(r);
			return pack;
		}

		public Msg ToMsg()
		{
			MemoryStream stm = new MemoryStream();
			Write(new BinaryWriter(stm));
			Msg.Data = stm.ToArray();
			//!!!Msg.Data = UnetUtils.ToArray(this);
			return Msg;
		}
	}

	public class OneTTLPacket : Packet
	{
		public OneTTLPacket()
		{
			Msg.TTL = 1;
		}
	}
  
	public class PingPacket : OneTTLPacket
	{
		public Guid Guid;
		public IPEndPoint TcpEndPoint, UdpEndPoint;

		public PingPacket()
		{
			Guid = UnetMan.Instance.MyGuid;
			TcpEndPoint = UnetMan.Instance.TUClient.TcpClient.PublicEndPoint;
			UdpEndPoint = UnetMan.Instance.TUClient.UdpClient.PublicEndPoint;
		}

		public override void Process()
		{
			if (Msg.PeerFrom != null)
			{
				if (TcpEndPoint != null)
					Msg.PeerFrom.TcpEndPoint = TcpEndPoint;
				if (UdpEndPoint != null)
					Msg.PeerFrom.UdpEndPoint = UdpEndPoint;
				Msg.PeerFrom.ReActivate();
			}
			else
			{
				IPEndPoint uep = UdpEndPoint!=null
					&& (UnetMan.Instance.TUClient.UdpClient.PublicEndPoint!=null || (UnetMan.Instance.Proxy is Socks5Proxy))
					? UdpEndPoint : Msg.UdpEndPoint;
				(Msg.PeerFrom = UnetMan.Instance.AddLiveEntry(Guid,TcpEndPoint,uep)).ReActivate();
			}
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Guid = Ut.ReadGuid(r);
			TcpEndPoint = Ut.ReadEndPoint(r);
			UdpEndPoint = Ut.ReadEndPoint(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_PING);
			Ut.Write(w,Guid);
			Ut.Write(w,TcpEndPoint);
			Ut.Write(w,UdpEndPoint);
		}
	}

	public class TimePacket : Packet
	{
		DateTime Dt;

		public TimePacket()
		{
			Dt = DateTime.UtcNow;
		}

		public TimePacket(DateTime dt)
		{
			Dt = dt;
		}

		public override void Process()
		{
			UnetMan.Instance.UnetDateTime = Dt;
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Dt = Ut.ReadDateTime(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_TIME);
			Ut.Write(w,Dt);
		}

	}

	public class QueryTimePacket : Packet
	{
		public override void Process()
		{
			Msg.PeerFrom.Send(new TimePacket());
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_QUERYTIME);
		}
	}

	public class VersionPacket : Packet
	{
		string Version;

		public VersionPacket()
		{
			Version = Dns.GetHostName()+": "+Assembly.GetCallingAssembly().GetName().Version.ToString();
		}

		public override void Process()
		{
			Logger.Log("Version: {0}",Version);
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Version = r.ReadString();
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_VERSION);
			w.Write(Version);
		}
	}

	public class QueryVersionPacket : Packet
	{
		public override void Process()
		{
			UnetMan.Instance.Relay(new VersionPacket());
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_QUERYVERSION);
		}
	}

	public class StatPacket : Packet
	{
		public string Stat;

		public override void Process()
		{
			Logger.Log("Statistics\n==========================\n{0}",Stat);
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			Stat = r.ReadString();
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_STAT);
			w.Write(Stat);
		}
	}

	public class QueryStatPacket : Packet
	{
		public override void Process()
		{
			ArrayList ar = new ArrayList();
			ar.Add(string.Format("Host:\t{0}\t{1}",Dns.GetHostName(),UnetMan.Instance.MyGuid));
			ar.Add(string.Format("Ver:\t{0}",Assembly.GetCallingAssembly().GetName().Version.ToString()));
			UnetMan um = UnetMan.Instance;
			lock (um.PeerMan)
			{
				ar.Add(string.Format("Peers:\t{0}",um.PeerMan.Peers.Count));
				ar.Add(string.Format("Active:\t{0}",um.PeerMan.ActivePeers.Count));
				foreach (Peer peer in um.PeerMan.ActivePeers)
					ar.Add(string.Format("  {0}",peer));
			}
			ar.Add(string.Format("Sent:\t{0} KB",um.SentBytes/1024));
			ar.Add(string.Format("Recv:\t{0} KB",um.ReceivedBytes/1024));
			try
			{
				Process process = System.Diagnostics.Process.GetCurrentProcess();
				ar.Add(string.Format("Paged Memory:\t{0} MB",process.PagedMemorySize64/(1024*1024)));
				ar.Add(string.Format("Non-paged System Memory:\t{0} MB",process.NonpagedSystemMemorySize64/(1024*1024)));
				ar.Add(string.Format("Private Memory:\t{0} MB",process.PrivateMemorySize64/(1024*1024)));
				ar.Add(string.Format("Total Processor Time:\t{0}",process.TotalProcessorTime));
			}
			catch (Exception)
			{}
			StringBuilder sb = new StringBuilder();
			foreach (string s in ar)
				sb.Append(s+"\n");
			StatPacket sp = new StatPacket();
			sp.Stat = sb.ToString();
			Logger.Log(sp.Stat);
			UnetMan.Instance.Relay(sp);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_QUERYSTAT);
		}
	}

	public class PeerPacket : OneTTLPacket
	{
		public Guid ID;
		public IPEndPoint TcpEndPoint,
			UdpEndPoint;

		public PeerPacket()
		{
		}

		public PeerPacket(Guid id, IPEndPoint t, IPEndPoint u)
		{
			ID = id;
			TcpEndPoint = t;
			UdpEndPoint = u;
		}

		public override void Process()
		{
			//!!!Logger.Log("Peer Packet {0} {1}", TcpEndPoint,UdpEndPoint);
			UnetMan.Instance.AddLiveEntry(ID,TcpEndPoint,UdpEndPoint);
		}

		public override void Read(BinaryReader r)
		{
			base.Read(r);
			ID = Ut.ReadGuid(r);
			TcpEndPoint = Ut.ReadEndPoint(r);
			UdpEndPoint = Ut.ReadEndPoint(r);
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_PEER);
			Ut.Write(w,ID);
			Ut.Write(w,TcpEndPoint);
			Ut.Write(w,UdpEndPoint);
		}
	}

	public class QueryPeersPacket : OneTTLPacket
	{
		public override void Process()
		{
			lock (UnetMan.Instance.PeerMan)
				foreach (Peer peer in UnetMan.Instance.PeerMan.Peers)
					if (!peer.Bad && (peer.TcpEndPoint!=null || peer.UdpEndPoint!=null))
						UnetMan.Instance.Relay(new PeerPacket(peer.ID,peer.TcpEndPoint,peer.UdpEndPoint));
		}

		public override void Write(BinaryWriter w)
		{
			w.Write(PACKET_QUERYPEERS);
		}
	}


}