using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Collections;

using Utils;

namespace ProxyUtils
{
	public class ConnectException : Exception
	{
	}

  public enum ProxyType
  {
		Default = 0,
    Direct  = 1,
    Http   = 2,
    Socks4 = 4,
    Socks5 = 5
  }

  public abstract class HostAddress
  {
  }

#region Exceptions

  public class ProxyException : Exception
  {
    public ProxyException()
    {}

    public ProxyException(string s)
      :base(s)
    {}
  }

  public class Socks5Exception : ProxyException
  {
    public Socks5Exception(string s)
      :base("SOCKS5: "+s)
    {}
  }

  public class Socks5GeneralFailureException : Socks5Exception
  {
    public Socks5GeneralFailureException()
      :base("General Failure")
    {}
  }

  public class Socks5NotAllowedByRulesetException : Socks5Exception
  {
    public Socks5NotAllowedByRulesetException()
      :base("Connection Not Allowed By Ruleset")
    {}
  }

  public class Socks5NetworkUnreachableException : Socks5Exception
  {
    public Socks5NetworkUnreachableException()
      :base("Network Unreachable")
    {}
  }
  
  public class Socks5HostUnreachableException : Socks5Exception
  {
    public Socks5HostUnreachableException()
      :base("Host Unreachable")
    {}
  }
  
  public class Socks5RefusedException : Socks5Exception
  {
    public Socks5RefusedException()
      :base("Connection Refused")
    {}
  }

  public class Socks5TTLExpiredException : Socks5Exception
  {
    public Socks5TTLExpiredException()
      :base("TTL Expired")
    {}
  }

  public class Socks5CommandNotSupportedException : Socks5Exception
  {
    public Socks5CommandNotSupportedException()
      :base("Command Not Supported")
    {}
  }

  public class Socks5AddressTypeNotSupportedException : Socks5Exception
  {
    public Socks5AddressTypeNotSupportedException()
      :base("Address Type Not Supported")
    {}
  }


#endregion


  public class DomainHostAddress : HostAddress
  {
    public string Domain;

    DomainHostAddress()
    {}

    public DomainHostAddress(string domain)
    {
      Domain = domain;
    }

    public override string ToString()
    {
      return Domain;
    }
  }

  public class Ip4HostAddress : HostAddress
  {
    public UInt32 IP4;

    public Ip4HostAddress()
    {}

    public Ip4HostAddress(UInt32 ip4)
    {
      IP4 = ip4;
    }

    public Ip4HostAddress(IPAddress addr)
    {
      byte[] ar = addr.GetAddressBytes();
      IP4 = (UInt32)(ar[0]|(ar[1]<<8)|(ar[2]<<16)|(ar[3]<<24));
    }

    public IPAddress ToIPAddress()
    {
      return new IPAddress(IP4);
    }

    public override string ToString()
    {
      return ToIPAddress().ToString();
    }
  }

  public class Ip6HostAddress : HostAddress
  {
    public byte[] IP6;

    public Ip6HostAddress()
    {}

    public Ip6HostAddress(byte[] ip6)
    {
      IP6 = (byte[])ip6.Clone();
    }

    public override string ToString()
    {
      Ut.Throw();
      return "";//!!!
    }
  }

  public class HostPort
  {
    public HostAddress HostAddress;
    public UInt16 Port;

    public HostPort()
    {}

    public HostPort(EndPoint ep)
    {
      IPEndPoint ipep = (IPEndPoint)ep;
      HostAddress = new Ip4HostAddress(ipep.Address);//!!!
      Port = (ushort)ipep.Port;
    }

    public override string ToString()
    {
      return HostAddress is Ip6HostAddress ?  HostAddress.ToString()+"."+Convert.ToString(Port)
        : HostAddress.ToString()+":"+Convert.ToString(Port);
    }

    public IPEndPoint GetEndPoint()
    {
      IPAddress ipaddr;
      if (HostAddress is DomainHostAddress)
        ipaddr = Dns.GetHostEntry(HostAddress.ToString()).AddressList[0];
      else
        ipaddr = ((Ip4HostAddress)HostAddress).ToIPAddress();
      return new IPEndPoint(ipaddr,Port);
    }
  }

  public abstract class Proxy
  {
    public static Proxy DefaultProxy;

    public HostPort HostPort;
    public string Login = "",
      Password = "";
    
    protected IPEndPoint endPoint;

    public IPEndPoint EndPoint
    {
      get
      {
        if (endPoint == null)
          endPoint = HostPort.GetEndPoint();
        return endPoint;
      }
    }

    public abstract void ConnectHTTPGet(TcpClient client, HostPort hp, string[] headers);

    public static void WriteHTTPGet(Stream stm, HostPort hp, string[] headers)
    {
      StreamWriter w = new StreamWriter(stm);
      w.WriteLine("POST http://{0} HTTP/1.1",hp);
      foreach (string line in headers)
        w.WriteLine(line);//!!!
      w.WriteLine("Cache-control: no-store, no-cache");//!!!
      w.WriteLine();
      w.Flush();
    }
  }

  public class Socks4Proxy : Proxy
  {
    void WriteSZ(BinaryWriter w, string s)
    {
      w.Write(s.ToCharArray());
      w.Write((byte)0);
    }

    IPEndPoint ReadReply(BinaryReader r)
    {
      byte[] ar = r.ReadBytes(8);
//!!! WinGaet returns 0      if (ar[0] != 4)
//        throw new ProxyException();
      switch (ar[1])
      {
        case 90: break;
        default:
          throw new ProxyException();
      }
      return new IPEndPoint((UInt32)((ar[4]<<24)|(ar[5]<<16)|(ar[6]<<8)|ar[7]),(ar[2]<<8)|ar[3]);
    }

    public void ConnectAndAuthenticate(TcpClient client, HostPort hp)
    {
      client.Connect(EndPoint);
			MemoryStream ms = new MemoryStream();
			BinaryWriter w = new BinaryWriter(ms);
      Stream stm = client.GetStream();
      BinaryWriter sw = new BinaryWriter(stm);
      BinaryReader r = new BinaryReader(stm);
      w.Write((byte)4);
      w.Write((byte)1);
      w.Write(IPAddress.HostToNetworkOrder((short)hp.Port));
      if (hp.HostAddress is DomainHostAddress)
      {
        DomainHostAddress dha = (DomainHostAddress)hp.HostAddress;
        UInt32 fakedIP = 0x01000000;
        w.Write(fakedIP);
        WriteSZ(w,Password);
        WriteSZ(w,dha.Domain);
      }
      else if (hp.HostAddress is Ip4HostAddress)
      {
        Ip4HostAddress ha = (Ip4HostAddress)hp.HostAddress;
        w.Write(ha.IP4);
        WriteSZ(w,Password);
      }
			sw.Write(ms.ToArray());
      ReadReply(r);
    }

    public override void ConnectHTTPGet(TcpClient client, HostPort hp, string[] headers)
    {
      ConnectAndAuthenticate(client,hp);
      WriteHTTPGet(client.GetStream(),hp,headers);
    }
  }

  public class TcpClientEx : TcpClient
  {
    public TcpClientEx()
    {}

    public TcpClientEx(Socket sock)
    {
      Client = sock;
    }

    public IPEndPoint GetLocalEndPoint()
    {
      return (IPEndPoint)Client.LocalEndPoint;
    }

    public IPEndPoint GetRemoteEndPoint()
    {
      return (IPEndPoint)Client.RemoteEndPoint;
    }

    public void CloseEx()
    {
      Client.Close();//!!!
    }
  }

  public class Socks5Bind : TcpClientEx
  {
    public IPEndPoint EndPoint;
  }

  internal class BindAddressDeterminer : IThreadable
  {
    Socks5UdpBind Bind;

    internal BindAddressDeterminer(Socks5UdpBind bind)
    {
      Bind = bind;
    }

    protected override void VExecute()
    {
      Socks5Bind bind = Bind.Proxy.Bind(Bind.LocEP);//!!!
      if (bind.EndPoint.Address.Equals(IPAddress.Any))
        Bind.BindAddress = Bind.Proxy.HostPort.GetEndPoint().Address;
      else
        Bind.BindAddress = bind.EndPoint.Address;
      Bind.SendTo(Bind.RandomArray,new IPEndPoint(Bind.LocEP.Address,((IPEndPoint)Bind.ReceiveSocket.LocalEndPoint).Port));
    }
  }

  public class Socks5UdpBind : IThreadable, IExceptable
  {
//!!!    public bool DetermineBindAddress;
    public IPAddress BindAddress;
    internal Socks5Proxy Proxy;
    internal TcpClientEx udpAssociationClient;
    public byte[] RandomArray;
    public Socket ReceiveSocket;
//!!!    BindAddressDeterminer BindAddressDeterminer;
    internal IPEndPoint LocEP;
    ThreadMan ThreadMan = new ThreadMan();
    public IPEndPoint UdpEndPoint;
    public Socket UdpSocket;

    public Socks5UdpBind()
    {
      Random random = new Random((int)DateTime.Now.Ticks);
      RandomArray = new byte[16];
      for (int i=0; i<RandomArray.Length; i++)
        RandomArray[i] = (byte)random.Next();
    }

    public override void Stop()
    {
      base.Stop();
      ThreadMan.Stop();
      try
      {
        udpAssociationClient.GetStream().Close();
        udpAssociationClient.Close();
      }
      catch (Exception)
      {}
      udpAssociationClient = null;
    }

    protected override void VExecute()
    {
      Proxy.ConnectAndAuthenticate(udpAssociationClient = new TcpClientEx());
      LocEP = udpAssociationClient.GetLocalEndPoint();
      Stream stm = udpAssociationClient.GetStream();
      BinaryWriter w = new BinaryWriter(stm);
      UdpEndPoint = Proxy.Cmd(udpAssociationClient,3,new IPEndPoint(0,0)).GetEndPoint();
     
        /*!!!        foreach (IPEndPoint ep in locs)
                  if (Util.IsGlobal(ep))
                  {
                    lep.Address = ep.Address;
                    goto found;
                  }
                foreach (IPEndPoint ep in locs)
                  if (new Ip4HostAddress(ep.Address).IP4 != 0x0100007F)
                  {
                    lep.Address = ep.Address;
                    goto found;
                  }
              found:*/
        //!!!        sw.Write(lep.Address.GetAddressBytes());//!!!
        //!!!        sw.Write(IPAddress.HostToNetworkOrder((Int16)lep.Port));
        //!!!        w.Write(s.ToArray());        
        //      }

        //!!!      byte[] ar = {5,3,0,1,0,0,0,0,0,0};
        //!!!      w.Write(ar);
        //!!!      UdpEndPoint = (IPEndPoint)ReadReply(r).GetEndPoint();

/*!!!      if (DetermineBindAddress && BindAddressDeterminer==null)
        ThreadMan.Add(BindAddressDeterminer=new BindAddressDeterminer(this));*/
      Proxy.ReadReply(new BinaryReader(udpAssociationClient.GetStream()));
    }

    public void SendTo(byte[] ar, IPEndPoint ep)
    {
      if (UdpEndPoint == null)
        return;
      UdpSocket.SendTo(Socks5Proxy.CreatePacket(ar,ep),UdpEndPoint);
    }

    public event ExceptionHandler OnException;

    void IExceptable.ExceptionRaised(Exception e)
    {
      if (OnException != null)
        OnException(e);
    }
  }

  public class Socks5Proxy : Proxy , IDisposable
  {
    public IPEndPoint ProxyEndPoint;
    ThreadMan ThreadMan = new ThreadMan();

    public Socks5Proxy()
    {}

    public Socks5Proxy(IPEndPoint ep)
    {
      endPoint = ProxyEndPoint = ep;
    }

    void IDisposable.Dispose()
    {
      ThreadMan.Stop();
    }

    public Socks5UdpBind AssociateUDP(Socket sock, bool determineBindAddress)
    {
      Socks5UdpBind bind = new Socks5UdpBind();
      bind.UdpSocket = sock;
      bind.ReceiveSocket = sock;
      bind.SleepTime = 2000;
      bind.Proxy = this;
//!!!      bind.DetermineBindAddress = determineBindAddress;
      ThreadMan.Add(bind);
      return bind;
    }

    public Socks5UdpBind AssociateUDP(Socket sock)
    {
      return AssociateUDP(sock,false);
    }

    public static HostPort ReadUdpRequest(BinaryReader r)
    {
      HostPort hp = new HostPort();
      byte[] header = r.ReadBytes(3);
      switch (r.ReadByte())
      {
        case 1:
          hp.HostAddress = new Ip4HostAddress(r.ReadUInt32());
          break;
        case 3:
          byte len = r.ReadByte();
          hp.HostAddress = new DomainHostAddress(new string(r.ReadChars(len)));
          break;
        case 4:
          hp.HostAddress = new Ip6HostAddress(r.ReadBytes(16));
          break;
        default:
          Ut.Throw();
          break;
      }
      hp.Port = (ushort)IPAddress.NetworkToHostOrder(r.ReadInt16());
      return hp;
    }

    public HostPort ReadReply(BinaryReader r)
    {
      HostPort hp = new HostPort();
      byte[] header = r.ReadBytes(4);
      if (header[0] != 5)
        throw new Socks5Exception("Invalid format");
      switch (header[1])
      {
        case 0: break;
        case 1: throw new ProxyException(); //!!!throw new Socks5GeneralFailureException();
        case 2: throw new Socks5NotAllowedByRulesetException();
        case 3: throw new Socks5NetworkUnreachableException();
        case 4: throw new Socks5HostUnreachableException();
        case 5: throw new Socks5RefusedException();
        case 6: throw new Socks5TTLExpiredException();
        case 7: throw new Socks5CommandNotSupportedException();
        case 8: throw new Socks5AddressTypeNotSupportedException();
        default:
          throw new Socks5Exception("Unknown");
      }
      switch (header[3])
      {
        case 1:
          hp.HostAddress = new Ip4HostAddress(r.ReadUInt32());
          break;
        case 3:
          byte len = r.ReadByte();
          hp.HostAddress = new DomainHostAddress(new string(r.ReadChars(len)));
          break;
        case 4:
          hp.HostAddress = new Ip6HostAddress(r.ReadBytes(16));
          break;
        default:
          throw new Socks5Exception("Invalid format");
      }
      hp.Port = (ushort)IPAddress.NetworkToHostOrder(r.ReadInt16());
      return hp;
    }

    public void ConnectAndAuthenticate(TcpClient client)
    {
      client.Connect(EndPoint);
      Stream stm = client.GetStream();
      BinaryWriter w = new BinaryWriter(stm);
      BinaryReader r = new BinaryReader(stm);
      byte[] ar = {5,2,0,2};
      w.Write(ar);
      if (r.ReadByte() != 5)
        Ut.Throw();
      byte b = r.ReadByte();
      switch (b)
      {
        case 0:
          break;
        case 2:
          w.Write((byte)1);
          w.Write((byte)Login.Length);
          w.Write(Login.ToCharArray());
          w.Write((byte)Password.Length);
          w.Write(Password.ToCharArray());
          b = r.ReadByte();
          b = r.ReadByte();
          if (b != 0)
            throw new ProxyException();
          break;
        default:
          Ut.Throw();
          break;
      }
    }

    public static byte[] CreatePacket(byte[] ar, IPEndPoint ep)
    {
      MemoryStream stm = new MemoryStream();
      BinaryWriter w = new BinaryWriter(stm);
      w.Write((Int16)0);
      w.Write((byte)0);
      w.Write((byte)1);
      w.Write(ep.Address.GetAddressBytes());
      w.Write(IPAddress.HostToNetworkOrder((Int16)ep.Port));
      w.Write(ar);
      return stm.ToArray();
    }

/*!!!    public void SendTo(byte[] ar, IPEndPoint ep)
    {
      if (UdpEndPoint == null)
        return;
      UdpSocket.SendTo(CreatePacket(ar,ep),UdpEndPoint);
    }*/

    internal HostPort Cmd(TcpClient client, byte cmd, IPEndPoint ep)
    {
      MemoryStream s = new MemoryStream();
      BinaryWriter sw = new BinaryWriter(s);
      sw.Write(new byte[]{5,cmd,0,1});
      sw.Write(ep.Address.GetAddressBytes());//!!!
      sw.Write(IPAddress.HostToNetworkOrder((Int16)ep.Port));
      Stream stm = client.GetStream();
      new BinaryWriter(stm).Write(s.ToArray());
      return ReadReply(new BinaryReader(stm));
    }

    Socks5Bind CmdBind(Socks5Bind bind, IPEndPoint ep)
    {
      bind.EndPoint = Cmd(bind,2,ep).GetEndPoint();
      return bind;
    }

    public Socks5Bind Bind(IPEndPoint ep)
    {
      Socks5Bind bind = new Socks5Bind();
      try
      {
        ConnectAndAuthenticate(bind);
        return CmdBind(bind,ep);
      }
      catch (Exception)
      {
        bind.CloseEx();
        throw;
      }
    }

    public Socks5Bind Bind(int port)
    {
      Socks5Bind bind = new Socks5Bind();
      try
      {
        ConnectAndAuthenticate(bind);
        return CmdBind(bind,new IPEndPoint(bind.GetLocalEndPoint().Address,port));
      }
      catch (Exception)
      {
        bind.CloseEx();
        throw;
      }
    }

    public override void ConnectHTTPGet(TcpClient client, HostPort hp, string[] headers)
    {
      ConnectAndAuthenticate(client);
      Cmd(client,1,hp.GetEndPoint());

/*!!!
      Stream stm = client.GetStream();
      MemoryStream mstm = new MemoryStream();
      BinaryWriter w = new BinaryWriter(stm);
      BinaryReader r = new BinaryReader(stm);
      w.Write((byte)5);
      w.Write((byte)0);
      w.Write((byte)1);
      if (hp.HostAddress is Ip4HostAddress)
      {
        w.Write((byte)1);
        Ip4HostAddress ha = (Ip4HostAddress)hp.HostAddress;
        w.Write(ha.IP4);
      }
      else if (hp.HostAddress is Ip6HostAddress)
      {
        w.Write((byte)4);
        Ip6HostAddress ha = (Ip6HostAddress)hp.HostAddress;
        w.Write(ha.IP6);
      }
      else
      {
        w.Write((byte)3);
        string domain = ((DomainHostAddress)hp.HostAddress).Domain;
        w.Write((byte)domain.Length);
        w.Write(domain.ToCharArray());
      }
      w.Write(IPAddress.HostToNetworkOrder((short)hp.Port));
      ReadReply(r);*/
      WriteHTTPGet(client.GetStream(),hp,headers);      
    }
  }

	public class IeProxy : Proxy
	{
		public override void ConnectHTTPGet(TcpClient client, HostPort hp, string[] headers)
		{
			/*!!!R
			WebProxy wp = WebProxy.GetDefaultProxy();
			Uri uri = wp.Address;
			*/
			Uri uri = null;
			if (uri != null)
			{
				IPEndPoint ep = new IPEndPoint(IPAddress.Parse(uri.Host),uri.Port);
				client.Connect(ep);
				WriteHTTPGet(client.GetStream(),hp,headers);
			}
			else
			{
				client.Connect(hp.GetEndPoint());
				Proxy.WriteHTTPGet(client.GetStream(),null,headers);
			}
		}
	}

  public class HttpProxy : Proxy
  {
    public override void ConnectHTTPGet(TcpClient client, HostPort hp, string[] headers)
    {
      client.Connect(EndPoint);
      WriteHTTPGet(client.GetStream(),hp,headers);
    }
  }

  class TcpListenerThreadable : IThreadable
  {
    ProxyTcpClient ProxyTcpClient;

    public TcpListenerThreadable(ProxyTcpClient client)
    {
      ProxyTcpClient = client;
      SleepTime = 0;
    }

    public override void Stop()
    {
      base.Stop();
      ProxyTcpClient.TcpListener.Stop();
    }

    protected override void VExecute()
    {
			Socket sock;
			try
			{
				sock = ProxyTcpClient.TcpListener.AcceptSocket();//!!!
			}
			catch (SocketException)
			{
				return;
			}
      ProxyTcpClient.Incoming(new TcpClientEx(sock));
    }
  }  

  public class ProxyTcpClient : TcpClient
  {
    public Proxy Proxy;
    public event ExceptionHandler OnException;
    internal TcpListener TcpListener;
    ThreadMan ThreadMan = new ThreadMan();

    public delegate void IncomingHandler(TcpClientEx client);
    public event IncomingHandler OnIncoming;

    internal void Incoming(TcpClientEx client)
    {
      if (OnIncoming != null)
        OnIncoming(client);
    }

    public WebHeaderCollection ConnectGet(TcpClient tc, IPEndPoint ep, string[] headers)
    {
      if (Proxy != null)
        Proxy.ConnectHTTPGet(tc,new HostPort(ep),headers);
      else
      {
        tc.Connect(ep);
        Proxy.WriteHTTPGet(tc.GetStream(),null,headers);
      }
      Stream stm = tc.GetStream();
      string line = Ut.ReadOneLineFromStream(stm); //!!! verify returned code 200
      string[] ar = line.Split(new char[]{' ','\t'});
      if (ar[1] != "200")
        throw new ConnectException();
			return Ut.ReadHttpHeader(stm);
			/*!!!
      ArrayList lines = new ArrayList();
      while ((line=r.ReadLine()) != "")
        lines.Add(line);
      string[] res = new string[lines.Count];
      for (int i=0; i<lines.Count; i++)
        res[i] = (string)lines[i];
      return res;*/
    }

    public void Bind(int port)
    {
      TcpListener = new TcpListener(IPAddress.Any,port);
      TcpListener.Start();
      ThreadMan.Add(new TcpListenerThreadable(this));
    }

	public IPEndPoint PublicEndPoint
	{
		get
		{
			/*
			if (Proxy != null && (!(Proxy is IeProxy) || WebProxy.GetDefaultProxy().Address != null))
				return null;
			*/
			IPAddress a = Ut.GetGlobalIPAddress();
			if (a == null)
				return null;
			return new IPEndPoint(a, ((IPEndPoint)TcpListener.LocalEndpoint).Port);
		}
	}

    internal void ExceptionRaised(Exception e)
    {
      if (OnException != null)
        OnException(e);
    }

    public void SignalStop()
    {
      ThreadMan.SignalStop();
    }

    public void Join()
    {
      ThreadMan.Join();
    }
  }

  class UdpReceiveThreadable : IThreadable
  {
    ProxyUdpClient Client;
    byte[] RandomArray;

    internal UdpReceiveThreadable(ProxyUdpClient client, byte[] ra)
    {
      Client = client;
      RandomArray = ra;
      SleepTime = 0;
    }

    public override void Stop()
    {
      base.Stop();
      Client.Client.Close();
    }

    protected override void VExecute()
    {
      EndPoint epr = new IPEndPoint(IPAddress.Any,0);
      byte[] ar = new byte[65536];
      int len = Client.Client.ReceiveFrom(ar,ref epr);
      IPEndPoint ep = (IPEndPoint)epr;
      byte[] dar = new byte[len];
      Array.Copy(ar,dar,len);
      if (Ut.Equals(dar,RandomArray))
        Client.publicPort = ep.Port;
      else
      {
        if (Client.UdpBind!=null && Client.UdpBind.UdpEndPoint!=null && Client.UdpBind.UdpEndPoint.Equals(ep))
        {
          MemoryStream stm = new MemoryStream(dar);
          BinaryReader r = new BinaryReader(stm);
          epr = Socks5Proxy.ReadUdpRequest(r).GetEndPoint();
          dar = r.ReadBytes((int)(stm.Length-stm.Position));
        }
        Client.Received(dar,ep);
      }
    }
  }

  class MsgToSend
  {
    public IPEndPoint EndPoint;
    public byte[] Data;

    public MsgToSend(IPEndPoint ep, byte[] data)
    {
      EndPoint = ep;
      Data = data;
    }
  }  

  internal class UdpSender : IThreadable
  {
    internal ProxyUdpClient Client;

    internal Queue Queue = Queue.Synchronized(new Queue());

    protected override void VExecute()
    {
      while (Queue.Count != 0)
      {
        MsgToSend msg = (MsgToSend)Queue.Dequeue();
        try
        {
          if (Client.UdpBind != null && Ut.IsGlobal(msg.EndPoint.Address))
            Client.UdpBind.SendTo(msg.Data,msg.EndPoint);
          else
            Client.Client.SendTo(msg.Data,msg.EndPoint);
        }
        catch (Exception)
        {
          //e = e; //!!!
        }
      }
    }
  }

  public class ProxyUdpClient
  {
    internal Socket Client = new Socket(AddressFamily.InterNetwork,SocketType.Dgram,ProtocolType.Udp);
    public Proxy Proxy;
    public event ExceptionHandler OnException;

    ThreadMan ThreadMan = new ThreadMan();
    //!!!    IPEndPoint EndPoint;
    IPAddress publicAddress;
    internal int publicPort;    
    internal Socks5UdpBind UdpBind;

    public IPEndPoint PublicEndPoint
    {
      get
      {
        if (publicPort == 0)
          return null;
        IPAddress a = publicAddress;
        if (a == null)
          a = Ut.GetGlobalIPAddress();
        if (a == null)
          return null;
        return new IPEndPoint(a,publicPort);

/*!!!        if (UdpBind != null)
          publicAddress = UdpBind.BindAddress;*/
      }
    }

    public delegate void ReceivedHandler(byte[] ar, IPEndPoint ep);
    public event ReceivedHandler OnReceived;

    internal void Received(byte[] ar, IPEndPoint ep)
    {
      if (OnReceived != null)
        OnReceived(ar,ep);
    }

    internal void ExceptionRaised(Exception e)
    {
      if (OnException != null)
        OnException(e);
    }

    public void Bind(int port)
    {
      Client.Bind(new IPEndPoint(0,port));
      Socks5Proxy socks5 = Proxy as Socks5Proxy;
      byte[] ra = null;
      if (socks5 != null)
      {
        (UdpBind = socks5.AssociateUDP(Client,true)).OnException += new ExceptionHandler(ExceptionRaised);
        ra = UdpBind.RandomArray;
      }
      else
      {
        publicAddress = Ut.GetGlobalIPAddress();
        publicPort = ((IPEndPoint)Client.LocalEndPoint).Port;
      }
      ThreadMan.Add(new UdpReceiveThreadable(this,ra));
    }

    UdpSender Sender = new UdpSender();

    public ProxyUdpClient()
    {
      Sender.Client = this;
      ThreadMan.Add(Sender);
    }

    public void SendTo(byte[] ar, IPEndPoint ep)
    {
      Sender.Queue.Enqueue(new MsgToSend(ep,ar));
      Sender.Thread.Interrupt();
    }    

    public void SignalStop()
    {
      ThreadMan.SignalStop();
    }

    public void Join()
    {
      ThreadMan.Join();
    }
  }

  public class TcpUdpClient
  {
    public ProxyTcpClient TcpClient = new ProxyTcpClient();
    public ProxyUdpClient UdpClient = new ProxyUdpClient();

    public void Bind(int port)
    {
      Logger.Log("TCP/UDP Binding");//!!!
      TcpClient.Bind(port);
      UdpClient.Bind(port);
//!!!      Unet.UnetMan.Log("TCP/UDP Binded");//!!!
    }

    public Proxy Proxy
    {
      get
      {
        return TcpClient.Proxy;
      }
      set
      {
        TcpClient.Proxy = value;
        UdpClient.Proxy = value;
      }
    }

    public void SignalStop()
    {
      TcpClient.SignalStop();
      UdpClient.SignalStop();
    }

    public void Join()
    {
      TcpClient.Join();
      UdpClient.Join();
    }

    public void Close()
    {
      SignalStop();
      Join();
    }
  }
}



