using System;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.Collections;
using System.Threading;
using System.IO;
using System.Text.RegularExpressions;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Diagnostics;
using System.Reflection;
using System.ServiceProcess;
using System.ServiceModel;
using System.ServiceModel.Channels;

using Utils;
using ProxyUtils;

namespace Unet
{
	[ServiceContract()]
	public interface IRelay
	{
		[OperationContract(IsOneWay = true)]
		void OnMsg(byte[] data);
	};


	public class PeerConnectException : Exception
	{
	}

	[Serializable]
	public class Peer
	{
		public Guid ID;
		public bool Bad;
		public bool My;
		public bool IsServer;
		public IPEndPoint TcpEndPoint, UdpEndPoint;
		[NonSerialized]
		public TcpConnection TcpConnection;

		public Peer()
		{ }

		public Peer(Guid id, IPEndPoint t, IPEndPoint u)
		{
			ID = id;
			TcpEndPoint = t;
			UdpEndPoint = u;
		}

		public string ToEndPointString()
		{
			if (TcpConnection != null)
				return TcpConnection.TcpClient.GetRemoteEndPoint().ToString();
			else
				return ToString();
		}

		public override string ToString()
		{
			string s = string.Format("Peer: {0}, {1} {2} {3}", TcpEndPoint, UdpEndPoint, (Bad ? "BAD" : "GOOD"), ID);
			try
			{
				if (TcpConnection != null && TcpConnection.TcpClient != null)
					s += "\n\t->" + TcpConnection.TcpClient.GetRemoteEndPoint().ToString();
			}
			catch (Exception)
			{ }
			return s;
		}

		public Int64 SentBytes,
					 ReceivedBytes;
		//!!!public bool Pinged;
		public DateTime TimeLastReceived, TimeLastSent;

		public void ReActivate()
		{
			Logger.Log("Reactivating {0}", this);
			Bad = false;
			PeerMan pm = UnetMan.Instance.PeerMan;
			lock (pm)
			{
				pm.ActivePeers.Remove(this);
				pm.ActivePeers.Insert(0, this);
				TimeLastReceived = DateTime.Now;
				//!!!        UnetMan.Log("Reactivated {0}",this);
			}
		}

		public void MakeGood()
		{
			if (Bad)
			{
				Logger.Log("Making good: {0}", this);
				Bad = false;
				PeerMan pm = UnetMan.Instance.PeerMan;
				lock (pm)
				{
					pm.Peers.Remove(this);
					pm.Peers.Insert(0, this);
					if (!UnetMan.Instance.UseTcp && TcpConnection == null && UnetMan.Instance.MyGuid != ID)
						ReActivate();
				}
			}
		}

		public void Activate()
		{
			if (ID == UnetMan.Instance.MyGuid)
				Ut.Throw();//!!!
			PeerMan pm = UnetMan.Instance.PeerMan;
			lock (pm)
			{
				if (pm.ActivePeers.Count > Params.MaxNumPeers)
					pm.ActivePeers.RemoveAt(pm.ActivePeers.Count - 1); //!!!
				ReActivate();
			}
		}

		public void DeactivateEx()
		{
			//!!!      UnetMan.Log("Deactivated {0}",this);
			PeerMan pm = UnetMan.Instance.PeerMan;
			lock (pm)
			{
				pm.ActivePeers.Remove(this);
				pm.Peers.Remove(this);
				pm.Peers.Add(this);
				/*!!!        if (pm.ActivePeers.Count < Params.NumPeers && !UnetMan.Instance.UseTcp)
						  foreach (Peer peer in pm.Peers)
							if (!pm.ActivePeers.Contains(peer))
							{
							  peer.ReActivate();
							  return;
							}*/
			}
		}

		public void Deactivate()
		{
			if (TcpConnection != null)
				try
				{
					TcpConnection.Thread.Interrupt();
				}
				catch (Exception)
				{ }
			else
				DeactivateEx();
		}

		public void Send(Msg msg)
		{
			MemoryStream stm = new MemoryStream();
			msg.Write(new BinaryWriter(stm));
			byte[] ar = stm.ToArray();
			SentBytes += ar.Length;
			UnetMan.Instance.SentBytes += (UInt32)ar.Length;
			if (TcpConnection != null)//!!! && TcpConnection.Connected)
			{
				//!!!Logger.Log("To TCP {0}",TcpEndPoint);
				try
				{
					BinaryWriter w = new BinaryWriter(TcpConnection.TcpClient.GetStream());
					w.Write(ar);
					//!!!          w.Flush();
				}
				catch (Exception e)
				{
					Logger.Log(e.Message);
					//!!!          TcpConnection = null;
				}
			}
			else if (UdpEndPoint != null)
			{
				Logger.Log("To UDP {0}", UdpEndPoint);
				UnetMan.Instance.TUClient.UdpClient.SendTo(ar, UdpEndPoint);
			}
			TimeLastSent = DateTime.Now;
			/*!!!
				  Socks5Proxy proxy = UnetMan.Instance.Proxy as Socks5Proxy;
				  if (proxy != null && proxy.UdpEndPoint != null)
					UnetMan.Instance.Sender.Queue.Enqueue(new MsgToSend(proxy.UdpEndPoint,proxy.CreatePacket(ar,EndPoint)));
				  UnetMan.Instance.Sender.Queue.Enqueue(new MsgToSend(EndPoint,ar)); //!!! any way event if has SOCKS5
				  UnetMan.Instance.Sender.Thread.Interrupt();*/
		}

		public void Send(Packet m)
		{
			//!!!      UnetMan.Log("Sending "+m.ToString());
			m.BeforeSend();
			Msg msg = m.ToMsg();
			lock (UnetMan.Instance.HtMsg)
				UnetMan.Instance.HtMsg[msg.Guid] = ++UnetMan.Instance.CurrentMsgNumber;
			Send(msg);
		}
	}

	public class TcpConnection : IThreadable
	{
		internal TcpClientEx TcpClient;
		internal bool Connected;
		int state; //!!!

		public TcpConnection(TcpClientEx tcpClient)
		{
			TcpClient = tcpClient;
			SleepTime = Params.ConnectionPause * 1000;
			UnetMan.Instance.ThreadMan.Add(this);
		}

		public override string ToString()
		{
			string s = "";
			try
			{
				s = TcpClient != null ? TcpClient.GetRemoteEndPoint().ToString() : "";
			}
			catch (Exception)
			{ }
			return string.Format("TcpConection {0}   State={1}", s, state);
		}

		public override void Stop()
		{
			Logger.Log("Stopping {0}", this);
			lock (this)
			{
				if (TcpClient != null)
					try
					{
						TcpClient.CloseEx();
					}
					catch (Exception)
					{ }
				TcpClient = null;
				base.Stop();
			}
		}

		Peer GuidFromHeaders(WebHeaderCollection headers, Peer peer)
		{
			string s = headers["Unet-Guid"];
			if (s == null)
				throw new PeerConnectException();
			Guid id = new Guid(s);
			if (id == UnetMan.Instance.MyGuid)
				throw new PeerConnectException();
			lock (UnetMan.Instance.PeerMan)
			{
				foreach (Peer p in UnetMan.Instance.PeerMan.ActivePeers)
					if (p.ID == id)
						throw new PeerConnectException();
				foreach (Peer p in UnetMan.Instance.PeerMan.Peers)
					if (p.ID == id)
					{
						peer = p;
						break;
					}
			}
			if (peer == null)
				peer = UnetMan.Instance.PeerMan.GetPeer(id, null, null, false);
			peer.ID = id;
			return peer;

			/*!!!
				  foreach (string s in headers)
					if (s.Substring(0,9) == "Unet-Guid")
					{
					  Guid id = new Guid(s.Split(' ')[1]);
					  if (id == UnetMan.Instance.MyGuid)
						throw new PeerConnectException();
					  lock (UnetMan.Instance.PeerMan)
					  {
						foreach (Peer p in UnetMan.Instance.PeerMan.ActivePeers)
						  if (p.ID == id)
							throw new PeerConnectException();
						foreach (Peer p in UnetMan.Instance.PeerMan.Peers)
						  if (p.ID == id)
						  {
							peer = p;
							break;
						  }
					  }
					  if (peer == null)
						peer = UnetMan.Instance.PeerMan.GetPeer(id,null,null,false);
					  peer.ID = id;
					  return peer;
					}
				  throw new PeerConnectException();
						*/
		}

		internal void ReceiveLoop(Stream stm, Peer peer)
		{
			Connected = true;
			try
			{
				peer.Activate();
				BinaryReader reader = new BinaryReader(stm);
				while (true)
				{
					Msg msg = new Msg();
					state = 7;
					msg.Read(reader);
					state = 8;
					msg.PeerFrom = peer;
					UnetMan.Instance.PreProcess(msg);
				}
			}
			catch (Exception e)
			{
				state = 6;
				Logger.Log("{0}  State={1}", e.Message, state);
				throw;
			}
			finally
			{
				state = 9;
				peer.DeactivateEx();
				state = 10;
			}
		}

		protected override void VExecute()
		{
			Peer peer = null;
			state = 1;
			try
			{
				if (UnetMan.Instance.UseTcp)
				{
					lock (UnetMan.Instance.PeerMan)
						foreach (Peer p in UnetMan.Instance.PeerMan.Peers)
							if (!p.Bad && !p.My && p.TcpEndPoint != null && p.TcpConnection == null
								&& (p.IsServer || !UnetMan.Instance.ServerMode))
							{
								(peer = p).TcpConnection = this;
								break;
							}
					lock (this)
					{
						if (bStop)
							Thread.Abort();
						TcpClient = new TcpClientEx();
					}
					state = 2;
					if (peer != null)
					{
						string[] headers = new string[] { "Unet-Guid: " + UnetMan.Instance.MyGuid.ToString() };
						Logger.Log("Connecting to {0}", peer.TcpEndPoint);
						try
						{
							state = 3;
							peer = GuidFromHeaders(UnetMan.Instance.TUClient.TcpClient.ConnectGet(TcpClient, peer.TcpEndPoint, headers), peer);
						}
						catch (Exception e)
						{
							state = 4;
							Logger.Log("Connecting failed: {0}\t{1}", peer.TcpEndPoint, e.Message);
							UnetMan.Instance.PeerMan.MakeLast(peer);
							throw;
						}
						state = 5;
						ReceiveLoop(TcpClient.GetStream(), peer);
					}
				}
			}
			catch (Exception)
			{
				if (peer != null)
					Logger.Log("Disconnected {0}", peer.TcpEndPoint);
				throw;
			}
			finally
			{
				if (peer != null)
					peer.TcpConnection = null;
				TcpClient = null;
				Connected = false;
			}
		}

		protected override void VExecuteNonLoop()
		{
			if (TcpClient != null)
			{
				StreamWriter w = null;
				try
				{
					Stream stm = TcpClient.GetStream();
					w = new StreamWriter(stm);
					Peer peer;
					try
					{
						string line = Ut.ReadOneLineFromStream(stm);
						peer = GuidFromHeaders(Ut.ReadHttpHeader(stm), null);
						lock (UnetMan.Instance.PeerMan)
						{
							if (peer.TcpConnection != null)
								throw new PeerConnectException();
							peer.TcpConnection = this;
						}
					}
					catch (Exception)
					{
						try
						{
							w.WriteLine("HTTP/1.0 400\r\n");
							w.Flush();
						}
						catch (Exception)
						{ }
						throw;
					}
					w.WriteLine("HTTP/1.0 200\r\nUnet-Guid: {0}\r\n", UnetMan.Instance.MyGuid);
					w.Flush();
					ReceiveLoop(stm, peer);
				}
				catch (Exception e)
				{
					Logger.Log("Non-Loop {0}   State={1}", e.Message, state);
					throw;
				}
				finally
				{
					Logger.Log("Exiting Non-loop");
					try
					{
						TcpClient.CloseEx();
					}
					catch (Exception)
					{ }
					Connected = false;
					Logger.Log("Exited Non-loop");
				}
			}
			else
				base.VExecuteNonLoop();
		}
	}

	public class PeerMan
	{
		public ArrayList Peers = new ArrayList(),
						 ActivePeers = new ArrayList();

		public void MakeAllGood()
		{
			Logger.Log("Making All Good");
			lock (this)
			{
				for (int i = Peers.Count - 1; i >= 0; i--)
				{
					Peer peer = (Peer)Peers[i];
					if (peer.Bad && (DateTime.Now - peer.TimeLastReceived) > TimeSpan.FromSeconds(Params.RemovePeerPeriod))
					{
						Logger.Log("Removing {0}", peer);
						Peers.Remove(peer);
					}
				}
				foreach (Peer peer in Peers)
					peer.Bad = false;
			}
		}

		public void MakeLast(Peer peer)
		{
			peer.Bad = true;
			lock (this)
			{
				Peers.Remove(peer);
				Peers.Add(peer);
			}
		}

		public Peer GetPeerByUdp(IPEndPoint u)
		{
			if (u != null)
				lock (this)
					foreach (Peer peer in Peers)
						if (u.Equals(peer.UdpEndPoint))
							return peer;
			return null;
		}

		public Peer GetPeer(Guid id, IPEndPoint t, IPEndPoint u, bool change)
		{
			lock (this)
			{
				foreach (Peer peer in Peers)
					if (peer.ID == id)
					{
						if (change)
						{
							peer.TcpEndPoint = t;
							peer.UdpEndPoint = u;
						}
						return peer;
					}
				if (Peers.Count >= Params.MaxNumPeers)
				{
					Peers.RemoveRange(Params.MaxNumPeers, Peers.Count - Params.MaxNumPeers);
					for (int i = Peers.Count - 1; i >= 0; i--)
						if (((Peer)Peers[i]).Bad)
							Peers.RemoveAt(i);
				}
				Peer p = new Peer(id, t, u);
				foreach (IPEndPoint mep in UnetMan.Instance.MyEndPoints)
					if (mep.Equals(t))
					{
						p.My = true;
						break;
					}
				Peers.Add(p);
				return p;
			}
		}
	}

	internal class BroadcastClient : UdpClient
	{
		internal BroadcastClient()
		{
			//!!!      Client.SetSocketOption(SocketOptionLevel.Socket,SocketOptionName.ReuseAddress,1);
			//!!!      IPEndPoint ep = new IPEndPoint(0,UnetMan.Instance.Port);
			Client.Bind(new IPEndPoint(0, 0));
		}

		internal void Broadcast()
		{
			Logger.Log("Broadcasting");
			MemoryStream stm = new MemoryStream();
			BinaryWriter w = new BinaryWriter(stm);
			w.Write((byte)255);
			ArrayList myEndPoints = UnetMan.Instance.MyEndPoints;
			w.Write(myEndPoints.Count);
			foreach (IPEndPoint ep in myEndPoints)
				Ut.Write(w, ep);
			byte[] ar = stm.ToArray();
			UInt16 port = Params.DefaultPort;
			for (int i = 0; i < Params.NumberOfAlternatePorts; i++)
				Send(ar, ar.Length, new IPEndPoint(IPAddress.Parse("255.255.255.255"), port + i));
			Close();
			Logger.Log("Broadcasted");
		}
	}

	/*!!!  internal class Sender : IThreadable
	  {
		internal Queue Queue = Queue.Synchronized(new Queue());

		protected override void VExecute()
		{
		  while (Queue.Count != 0)
			try
			{
			  MsgToSend msgToSend = (MsgToSend)Queue.Dequeue();
			  UnetMan.Instance.TUClient.UdpClient.SendTo(msgToSend.Data,msgToSend.EndPoint);
			}
			catch (Exception)
			{}
		}
	  }*/

	internal class MsgExecutor : IThreadable
	{
		internal MsgExecutor()
		{
			SleepTime = Params.PingPeriod * 1000;
		}

		protected override void VExecuteNonLoop()
		{
			while (!bStop)
			{
				lock (UnetMan.Instance.QueueMsg)
					Monitor.Wait(UnetMan.Instance.QueueMsg, SleepTime);
				while (UnetMan.Instance.QueueMsg.Count != 0)
					UnetMan.Instance.Process((Msg)UnetMan.Instance.QueueMsg.Dequeue());
				DateTime dtForPing = DateTime.Now - TimeSpan.FromMilliseconds(SleepTime),
				  dtForPong = DateTime.Now - TimeSpan.FromMilliseconds(SleepTime * 3);
				bool newConnected = UnetMan.Instance.TimeLastReceived > dtForPong;
				if (UnetMan.Instance.Connected != newConnected)
					UnetMan.Instance.ConnectionStateChanged(newConnected);
				PeerMan pm = UnetMan.Instance.PeerMan;
				lock (pm)
					for (int i = pm.ActivePeers.Count - 1; i >= 0; i--)
					{
						Peer peer = (Peer)pm.ActivePeers[i];
						if (peer.TimeLastReceived < dtForPong)
							peer.Deactivate();//!!!
						else if (peer.TimeLastReceived < dtForPing || peer.TimeLastSent < dtForPing)
							peer.Send(new PingPacket());
					}

			}
		}
	}

	internal class WebExecutor : IThreadable
	{
		internal WebExecutor()
		{
			SleepTime = 5000;
		}

		DateTime dtPrev;

		bool bDownloaded;

		void ReadFrom(string url)
		{
			bDownloaded = false;
			try
			{
				StreamReader r = new StreamReader(UnetMan.RequestUrl(url).GetResponseStream());
				for (string line; (line = r.ReadLine()) != null; )
				{
					Logger.Log("Adding Peer {0}", line);
					UnetMan.Instance.AddLiveEntry(Ut.ParseHostPort(line)).IsServer = true;
				}
				bDownloaded = true;
				Logger.Log("Readed from " + url);
			}
			catch (Exception e)
			{
				Logger.Log(e.Message);
				if (bStop)
					Thread.Abort();
			}
		}

		bool bBroadcasted;

		protected override void VExecute()
		{
			int port = ((IPEndPoint)UnetMan.Instance.TUClient.UdpClient.Client.LocalEndPoint).Port;
			ArrayList ar = new ArrayList(),
					  newMyEndPoints = new ArrayList();
			foreach (IPAddress a in Dns.GetHostEntry(Dns.GetHostName()).AddressList)
			{
				newMyEndPoints.Add(new IPEndPoint(a, port));
				foreach (IPEndPoint ep in UnetMan.Instance.MyEndPoints)
					if (ep.Address.Equals(a))
						goto found;
				ar.Add(a);
			found:
				;
			}
			UnetMan.Instance.MyEndPoints = newMyEndPoints;
			if (!bBroadcasted)
			{
				bBroadcasted = true;
				new BroadcastClient().Broadcast();
			}
			bool bNewEP = ar.Count != 0;
			if (bNewEP)
				Logger.Log("New IP found");
#if DEBUG
			//!!!			UnetMan.Instance.AddLiveEntry(new IPEndPoint(IPAddress.Parse("192.168.38.4"),Params.DefaultPort));
#endif
			if (bNewEP && !bDownloaded || (DateTime.Now - dtPrev) > TimeSpan.FromSeconds(Params.WebRequestPeriod))
			{
				UnetMan.Instance.PeerMan.MakeAllGood();
				dtPrev = DateTime.Now;
				ReadFrom("http://www.ufasoft.com/p2p/peers.txt");
#if !DEBUG
        ReadFrom("http://www.p2p-messenger.net/p2p/peers.txt");
#endif
			}
			foreach (IPAddress a in ar)
			{
				if (!Ut.IsGlobal(a))
					continue;
				Logger.Log("Publishing {0}", a);
				IPEndPoint pubTcpEP = UnetMan.Instance.TUClient.TcpClient.PublicEndPoint,
						   pubUdpEP = UnetMan.Instance.TUClient.UdpClient.PublicEndPoint;
				if (pubTcpEP == null || pubUdpEP == null)
					continue;
				string url = (new Random((int)DateTime.Now.Ticks).Next() < Int32.MaxValue / 2)
				  ? "http://www.ufasoft.com"
				  : "http://www.p2p-messenger.net";
#if DEBUG
				url = "http://www.ufasoft.com";
#endif
				url = string.Format("{0}/p2p/addpeer.cgi?{1}", url, pubTcpEP);
				UnetMan.Instance.Relay(new PeerPacket(UnetMan.Instance.MyGuid, pubTcpEP, pubUdpEP));
				try
				{
					UnetMan.RequestUrl(url);
				}
				catch (WebException e)
				{
					Logger.Log(e.Message);
					if (bStop)
						Thread.Abort();
				}
			}
		}
	}

	[ServiceBehavior(InstanceContextMode = InstanceContextMode.Single)]
	public class UnetMan : IThreadable, IRelay
	{
		public Proxy Proxy;
		public PeerMan PeerMan = new PeerMan();
		public ArrayList Services = new ArrayList();
		public Hashtable HtMsg = new Hashtable();
		internal Queue QueueMsg = Queue.Synchronized(new Queue());
		MsgExecutor MsgExecutor = new MsgExecutor();
		//!!!    internal Sender Sender;

		//!!!    public ArrayList TcpConnections = new ArrayList();

		public DateTime UnetDateTime,
		  LocalDateTime;

		static public UnetMan Instance;
		public TcpUdpClient TUClient;
		public Guid MyGuid = Guid.NewGuid();

		public UnetMan()
		{
			Instance = this;
			UnetDateTime = LocalDateTime = DateTime.Now;
		}

		//!!!    public UInt16 Port;
		//!!!    public Socket UdpSocket;
		//!!!    public TcpListener TcpListener;
		public ThreadMan ThreadMan = new ThreadMan();


		/*!!!    bool stopUdpClient;


			Thread udpAssociationThread;*/

		/*!!!    public void StartUdpProxy()
			{
		//!!!      stopUdpClient = false;
			  Socks5Proxy proxy = Proxy as Socks5Proxy;
			  if (proxy != null)
				proxy.ConnectUDP(UdpSocket);
				//!!!(udpAssociationThread = new Thread(new ThreadStart(UdpAssociationExecute))).Start();
			}*/

		internal static WebResponse RequestUrl(string url)
		{
			Logger.Log("Requesting {0}", url);
			HttpWebRequest req = (HttpWebRequest)WebRequest.Create(url);
			req.UserAgent = "P2P";
			return req.GetResponse();
		}

		public void StopUdpProxy()
		{
			Socks5Proxy proxy = Proxy as Socks5Proxy;
			if (proxy != null)
				((IDisposable)proxy).Dispose();
		}

		public DateTime TimeLastReceived;
		public UInt64 SentBytes, ReceivedBytes; //!!! need lock

		void OnReceived(byte[] ar, IPEndPoint ep)
		{
			if (MyEndPoints == null) //!!!
			{
				Logger.Log("MyEndPoints == null");
				return;
			}

			foreach (IPEndPoint mep in MyEndPoints)
				if (mep.Equals(ep))
					return;
			MemoryStream stm = new MemoryStream(ar);
			BinaryReader r = new BinaryReader(stm);
			byte b = r.ReadByte();
			if (b == 255)
			{
				Logger.Log("On 255");
				Msg msgPing = new PingPacket().ToMsg();
				lock (HtMsg)
					HtMsg[msgPing.Guid] = ++CurrentMsgNumber;
				MemoryStream mstm = new MemoryStream();
				BinaryWriter w = new BinaryWriter(mstm);
				msgPing.Write(w);
				byte[] buf = mstm.ToArray();
				try
				{
					ArrayList peers = new ArrayList();
					int count = r.ReadInt32();
					for (int i = 0; i < count; i++)
						peers.Add(Ut.ReadEndPoint(r));

					//!!!              foreach (IPEndPoint dep in (ArrayList)new BinaryFormatter().Deserialize(stm))
					foreach (IPEndPoint dep in peers)
					{
						foreach (IPEndPoint mep in UnetMan.Instance.MyEndPoints)
							if (mep.Equals(dep))
							{
								if (new Ip4HostAddress(dep.Address).IP4 == 0x0100007F)
									goto localFound;
								return;
							}
						try
						{
							UnetMan.Instance.TUClient.UdpClient.Client.SendTo(buf, dep);
						}
						catch (Exception)
						{
						}
					localFound:
						;
					}
				}
				catch (Exception)
				{ }
        finally //!!!
				{
					Logger.Log("Leave On 255");
				}
			}
			else
			{
				Msg msg = new Msg();
				try
				{
					msg.Read(r, b);
				}
				catch (Exception)
				{
					return;
				}
				msg.UdpEndPoint = ep;
				PreProcess(msg);
			}
		}

		void OnIncoming(TcpClientEx client)
		{
			Logger.Log("Accepted TCP from {0}", client.GetRemoteEndPoint());
			new TcpConnection(client);
			/*!!!      lock (UnetMan.Instance.TcpConnections)
					UnetMan.Instance.TcpConnections.Add(conn);*/
		}

		public delegate bool UseTcpHandler();
		public UseTcpHandler OnUseTcp;

		public delegate Proxy GetProxyHandler();
		public GetProxyHandler OnGetProxy;

		public void UpdateSettings()
		{
			if (Proxy != null)
				StopUdpProxy(); //!!!
			if (OnUseTcp != null)
				UseTcp = OnUseTcp();
			else
				UseTcp = true;
			if (OnGetProxy != null)
				Proxy.DefaultProxy = Proxy = OnGetProxy();
			if (TUClient != null)
				TUClient.Proxy = Proxy;
			//!!!        UnetMan.StartUdpProxy();
		}

		void OnException(Exception e)
		{
			Logger.Log(e.Message);
		}

		public UInt16 StartingPort = Params.DefaultPort;

		ServiceHost ServiceHost;
		IRelay Channel;

		public override void Start()
		{
			try
			{
				var svc = new ServiceController("PNRPsvc");
			}
			catch (Exception)
			{
				throw new ApplicationException("P2P services not installed, please go to Control Panel | Add/Remove programs | Add Windows components | Network components. And check Peer-to-Peer services");
			}

			ServiceHost = new ServiceHost(this);
			NetPeerTcpBinding binding = new NetPeerTcpBinding();
			binding.Security.Mode = SecurityMode.None;
			string address = "net.p2p://ufasoft.unet";
			ServiceHost.AddServiceEndpoint(typeof(IRelay), binding, address);
			ServiceHost.Open();
			binding = new NetPeerTcpBinding();
			binding.Security.Mode = SecurityMode.None;
			Channel = new ChannelFactory<IRelay>(binding, address).CreateChannel();
//!!!			Channel = new ChannelFactory<IRelay>("ChatEndpoint").CreateChannel();

			UpdateSettings();
			for (int port = StartingPort; true; port++)
				try
				{
					(TUClient = new TcpUdpClient()).Proxy = Proxy;
					TUClient.Bind(port);
					break;
				}
				catch (Exception)
				{
					TUClient.Close();
				}
			TUClient.UdpClient.OnReceived += new ProxyUdpClient.ReceivedHandler(OnReceived);
			TUClient.TcpClient.OnIncoming += new ProxyTcpClient.IncomingHandler(OnIncoming);

			TUClient.UdpClient.OnException += new ExceptionHandler(OnException);

			//!!!      StartUdpProxy();
			//!!!      ThreadMan.Add(Sender=new Sender());
			ThreadMan.Add(new WebExecutor());
			ThreadMan.Add(MsgExecutor);
			//!!!Thread.Sleep(100000);//!!!
			//!!!      ThreadMan.Add(new MyPeerFiller());
			//!!!        TcpConnections.Add(conn);
		}

		public override void Stop()
		{
			base.Stop();
			TUClient.SignalStop();
			ThreadMan.SignalStop();
			StopUdpProxy();
			//!!!      UdpSocket.Close();
		}

		public override void Join()
		{
			TUClient.Join();
			ThreadMan.Join();
		}

		internal ArrayList MyEndPoints = new ArrayList();

		bool m_serverMode;

		public bool ServerMode
		{
			get
			{
				return m_serverMode;
			}
			set
			{
				if (m_serverMode = value)
					StartingPort = 4777;
			}
		}

		int NumPeers
		{
			get
			{
				return ServerMode ? Params.NumPeers * 3 : Params.NumPeers;
			}
		}

		bool useTcp;
		internal bool UseTcp
		{
			get
			{
				return useTcp || (Proxy != null && !(Proxy is Socks5Proxy))
							  || (Proxy == null && UnetMan.Instance.TUClient.UdpClient.PublicEndPoint == null);
			}
			set
			{
				useTcp = value;
			}
		}

		public bool NeedToCreateConnections = true;

		void EnsureCreateConnections()
		{
			if (NeedToCreateConnections)
			{
				NeedToCreateConnections = false;
				for (int i = 0; i < NumPeers; i++)
					new TcpConnection(null);
			}
		}

		public Peer AddLiveEntry(IPEndPoint ep)
		{
			EnsureCreateConnections();
			Peer peer = PeerMan.GetPeerByUdp(ep);
			if (peer == null)
				peer = AddLiveEntry(Guid.NewGuid(), ep, ep);
			return peer;
		}

		public Peer AddLiveEntry(Guid id, IPEndPoint t, IPEndPoint u)
		{
			EnsureCreateConnections();
			Peer peer = PeerMan.GetPeerByUdp(u);
			if (peer == null)
				(peer = PeerMan.GetPeer(id, t, u, true)).MakeGood();
			else if (peer.Bad)
			{
				peer.ID = id;
				peer.TcpEndPoint = t;
			}
			return peer;
		}

		public DateTime GetTime()
		{
			return UnetDateTime + (DateTime.Now - LocalDateTime);
		}

		internal int CurrentMsgNumber;

		internal void PreProcess(Msg msg)
		{
			lock (HtMsg)
				if (!HtMsg.Contains(msg.Guid))
				{
					if (HtMsg.Count >= Params.LastMessages)
					{
						ArrayList ar = new ArrayList();
						foreach (DictionaryEntry de in HtMsg)
							if (CurrentMsgNumber - (int)de.Value > Params.LastMessages / 2)
								ar.Add(de.Key);
						foreach (object ob in ar)
							HtMsg.Remove(ob);
					}
					HtMsg[msg.Guid] = ++CurrentMsgNumber;
					/*!!!        if (msg.UdpEndPoint != null)
							  msg.PeerFrom = PeerMan.GetPeerByUdp(msg.UdpEndPoint);*/
					UnetMan.Instance.TimeLastReceived = DateTime.Now;
					lock (QueueMsg)
					{
						QueueMsg.Enqueue(msg);
						Monitor.Pulse(QueueMsg);
					}
					//!!!          MsgExecutor.Thread.Interrupt();
				}
		}

		public void OnMsg(byte[] data)
		{
			var stm = new MemoryStream(data);
			Msg msg = new Msg();
			msg.Read(new BinaryReader(stm));
			Process(msg);
		}

		void Relay(Msg msg)
		{
			lock (HtMsg)
				HtMsg[msg.Guid] = ++CurrentMsgNumber;

			MemoryStream stm = new MemoryStream();
			msg.Write(new BinaryWriter(stm));
			Channel.OnMsg(stm.ToArray());


			/*!!!
			lock (PeerMan)
				foreach (Peer peer in PeerMan.ActivePeers)
					if (msg.PeerFrom != peer)
						peer.Send(msg);
			 */
		 
		}

		public void Relay(Packet m)
		{
			Logger.Log("Sending {0}", m);
			m.BeforeSend();
			Relay(m.ToMsg());
		}

		public void Process(Msg msg)
		{
			if (msg.PeerFrom == null)
				msg.PeerFrom = PeerMan.GetPeerByUdp(msg.UdpEndPoint);
			if (msg.PeerFrom != null)
			{
				msg.PeerFrom.TimeLastReceived = DateTime.Now;
				UInt32 count = (UInt32)(msg.Data.Length + 18);//!!!
				msg.PeerFrom.ReceivedBytes += count;
				ReceivedBytes += count;
			}
			while (msg.Ver == 2)
			{
				try
				{
					Packet m = msg.ToPacket();
					if (m == null)
						break;
					m.Msg = msg;
					m.Process();
					m.Log();
				}
				catch (Exception e)
				{
					Logger.Log(e);//!!!
				}
				break;
			}
			if (msg.PeerFrom != null)
				msg.PeerFrom.MakeGood();
//!!!R			if (!msg.DisableRelay && --msg.TTL != 0)
			//!!!R				Relay(msg);
		}

		public bool Connected;

		public delegate void ConnectionStateChangedHandler(bool connected);
		public event ConnectionStateChangedHandler OnConnectionStateChanged;

		internal void ConnectionStateChanged(bool newConnected)
		{
			Connected = newConnected;
			if (OnConnectionStateChanged != null)
				OnConnectionStateChanged(Connected);
			if (Connected)
				Relay(new QueryPeersPacket());
		}

		public void ShowPeers(bool bAll)
		{
			int i = 0;
			lock (PeerMan)
				if (bAll)
					foreach (Peer peer in PeerMan.Peers)
						Logger.Log("{0}. {1}", ++i, peer);
				else
					foreach (Peer peer in PeerMan.ActivePeers)
						Logger.Log("{0}. {1}", ++i, peer);
		}

		public void QueryVersion()
		{
			Relay(new QueryVersionPacket());
		}

		public void QueryStat()
		{
			new QueryStatPacket().Process();//!!!
			Relay(new QueryStatPacket());
		}

		public void OnCommand(string s)
		{
			Regex re = new Regex(@"^/(\w+)");
			Match match = re.Match(s);
			if (match.Length == 0)
				Logger.Log("InvalidCommand");
			else
			{
				string cmd = match.Groups[1].Value.ToUpper();
				switch (cmd)
				{
					case "EXIT":
						//!!!AppClass.Exit();
						break;
					case "PEERS": ShowPeers(false); break;
					case "TIME":
						Logger.Log("Current time: {0}", GetTime());
						break;
					case "MYPORT":
						Logger.Log("My Port: {0}", ((IPEndPoint)TUClient.UdpClient.Client.LocalEndPoint).Port);
						break;
					case "QUERYTIME":
						Relay(new QueryTimePacket());
						break;
					case "QUERYVERSION": QueryVersion(); break;
					default:
						Logger.Log("Unknown command");
						break;
				}
			}
		}
	}
}